package com.atlassian.integrationtesting.ui;

import java.io.IOException;
import java.net.URL;
import java.util.concurrent.TimeUnit;

import com.atlassian.sal.api.ApplicationProperties;

import com.gargoylesoftware.htmlunit.AlertHandler;
import com.gargoylesoftware.htmlunit.BrowserVersion;
import com.gargoylesoftware.htmlunit.Page;
import com.gargoylesoftware.htmlunit.WebClient;
import com.gargoylesoftware.htmlunit.html.HtmlElement;
import com.gargoylesoftware.htmlunit.html.HtmlPage;
import com.google.common.base.Function;

import be.roam.hue.doj.Doj;

import static com.google.common.base.Preconditions.checkState;

/**
 * <p>An implementation of {@code UiTester} that can be configured by composition of functions.  This allows us to share
 * significant amounts of code across both axis of products and versions without getting into a hairy inheritance
 * hierarchy.</p>
 * 
 * <p>Clients are expected to provide mechanisms to log in to an application - both regular log in and via web sudo,
 * log out of an application, determine the username of the logged in user and whether a page is the log in page.</p>
 */
public final class CompositeUiTester implements UiTester
{
    public static final long DELAY_MILLIS = TimeUnit.SECONDS.toMillis(5);
    public static final long TIMEOUT = TimeUnit.SECONDS.toMillis(2);

    private final ApplicationProperties applicationProperties;
    private final WebClient webClient;

    private final Function<Login, HtmlPage> login; 
    private final Function<WebSudoLogin, HtmlPage> webSudoLogin; 
    private final Function<UiTester, Void> logout;
    private final Function<UiTester, String> getLoggedInUser;
    private final Function<UiTester, Boolean> isOnLogInPage;
    private final Function<Backup, Void> restore;
    
    private HtmlPage currentPage;
    private boolean loggedIn;

    CompositeUiTester(ApplicationProperties applicationProperties,
            Function<Login, HtmlPage> login, 
            Function<WebSudoLogin, HtmlPage> webSudoLogin, 
            Function<UiTester, Void> logout,
            Function<UiTester, String> getLoggedInUser,
            Function<UiTester, Boolean> isOnLogInPage,
            Function<Backup, Void> restore)
    {
        this.applicationProperties = applicationProperties;
        this.webClient = new WebClient(BrowserVersion.INTERNET_EXPLORER_7);
        webClient.setThrowExceptionOnScriptError(false);
        webClient.setAlertHandler(new AlertHandler()
        {
            public void handleAlert(Page page, String message)
            {
                System.out.println("ALERT: " + message);
            }
        });
        
        this.login = login;
        this.webSudoLogin = webSudoLogin;
        this.logout = logout;
        this.getLoggedInUser = getLoggedInUser;
        this.isOnLogInPage = isOnLogInPage;
        this.restore = restore;
    }

    /**
     * Information passed to the function to log a user in to the application.
     */
    public static final class Login
    {
        public final UiTester client;
        public final String username;
        
        public Login(UiTester client, String username)
        {
            this.client = client;
            this.username = username;
        }
    }
    
    /**
     * Information passed to the function to perform a web sudo log in to the application. 
     */
    public static final class WebSudoLogin
    {
        public final UiTester client;
        public final String password;
        
        public WebSudoLogin(UiTester client, String password)
        {
            this.client = client;
            this.password = password;
        }
    }
    
    /**
     * Information passed to the function to perform a restore operation from backup data.
     */
    public static final class Backup
    {
        public final UiTester client;
        public final URL data;
        public final String username;
        
        public Backup(UiTester client, URL data, String username)
        {
            this.client = client;
            this.data = data;
            this.username = username;
        }
        
        @Override
        public String toString()
        {
            return "Backup(" + data.toString() + ")";
        }
    }
    
    /* (non-Javadoc)
     * @see com.atlassian.integrationtesting.UiTester#getLoggedInUser()
     */
    public String getLoggedInUser()
    {
        if (!loggedIn)
        {
            return null;
        }

        return getLoggedInUser.apply(this);
    }

    /* (non-Javadoc)
     * @see com.atlassian.integrationtesting.UiTester#isOnLoginPage()
     */
    public boolean isOnLoginPage()
    {
        return isOnLogInPage.apply(this);
    }

    /* (non-Javadoc)
     * @see com.atlassian.integrationtesting.UiTester#destroy()
     */
    public void destroy()
    {
        webClient.closeAllWindows();
    }

    /* (non-Javadoc)
     * @see com.atlassian.integrationtesting.UiTester#getBaseUrl()
     */
    public String getBaseUrl()
    {
        return applicationProperties.getBaseUrl();
    }

    /* (non-Javadoc)
     * @see com.atlassian.integrationtesting.UiTester#logInAs(java.lang.String)
     */
    public void logInAs(String username)
    {
        if (loggedIn)
        {
            throw new IllegalStateException("already logged in");
        }

        if (currentPage == null)
        {
            gotoPage("");
        }

        currentPage = login.apply(new Login(this, username));

        currentPage = webSudoLogin.apply(new WebSudoLogin(this, username));
        
        loggedIn = true;
    }

    /* (non-Javadoc)
     * @see com.atlassian.integrationtesting.UiTester#logout()
     */
    public void logout()
    {
        if (!loggedIn)
        {
            return;
        }

        checkCurrentPage();
        logout.apply(this);
        loggedIn = false;
    }

    /* (non-Javadoc)
     * @see com.atlassian.integrationtesting.UiTester#gotoPage(java.lang.String)
     */
    public HtmlPage gotoPage(String relativePath)
    {
        try
        {
            currentPage = webClient.getPage(ensureTrailingSlash(getBaseUrl()) + relativePath);
            waitForAsyncEventsThatBeginWithinDefaultTime();
            return currentPage;
        }
        catch (IOException e)
        {
            throw new RuntimeException(e);
        }
    }

    /* (non-Javadoc)
     * @see com.atlassian.integrationtesting.UiTester#refreshPage()
     */
    public Page refreshPage()
    {
        try
        {
            currentPage = (HtmlPage) currentPage.refresh();
            waitForAsyncEventsThatBeginWithinDefaultTime();
            return currentPage;
        }
        catch (IOException e)
        {
            throw new RuntimeException(e);
        }
    }

    /* (non-Javadoc)
     * @see com.atlassian.integrationtesting.UiTester#getStyleClass(java.lang.String)
     */
    public String getStyleClass(String elementId)
    {
        HtmlElement element = currentPage.getElementById(elementId);
        checkState(element != null, "Cannot find element: " + elementId);
        return element.getAttribute("class");
    }

    /* (non-Javadoc)
     * @see com.atlassian.integrationtesting.UiTester#clickElementWithId(java.lang.String)
     */
    public void clickElementWithId(String id)
    {
        HtmlElement element = currentPage.getElementById(id);

        checkState(element != null, "Cannot find element: " + id);
        try
        {
            element.click();
            waitForAsyncEventsThatBeginWithinDefaultTime();
        }
        catch (IOException e)
        {
            throw new RuntimeException(e);
        }
    }
    
    private void checkCurrentPage()
    {
        checkState(currentPage != null, "No current page found");
    }

    protected final String ensureTrailingSlash(String url)
    {
        return url.endsWith("/") ? url : url + "/";
    }

    /* (non-Javadoc)
     * @see com.atlassian.integrationtesting.UiTester#waitForAsyncEventsToComplete()
     */
    public int waitForAsyncEventsToComplete()
    {
        return waitForAsyncEventsToComplete(TIMEOUT);
    }

    /* (non-Javadoc)
     * @see com.atlassian.integrationtesting.UiTester#waitForAsyncEventsToComplete(long)
     */
    public int waitForAsyncEventsToComplete(long timeoutMillis)
    {
        checkCurrentPage();
        return webClient.waitForBackgroundJavaScript(timeoutMillis);
    }

    /* (non-Javadoc)
     * @see com.atlassian.integrationtesting.UiTester#waitForAsyncEventsThatBeginWithinDefaultTime()
     */
    public int waitForAsyncEventsThatBeginWithinDefaultTime()
    {
        return waitForAsyncEventsThatBeginWithin(DELAY_MILLIS);
    }

    /* (non-Javadoc)
     * @see com.atlassian.integrationtesting.UiTester#waitForAsyncEventsThatBeginWithin(long)
     */
    public int waitForAsyncEventsThatBeginWithin(long delayMillis)
    {
        checkCurrentPage();
        return webClient.waitForBackgroundJavaScriptStartingBefore(delayMillis);
    }

    /* (non-Javadoc)
     * @see com.atlassian.integrationtesting.UiTester#getElementById(java.lang.String)
     */
    public HtmlElement getElementById(String id)
    {
        return currentPage.getHtmlElementById(id);
    }

    /* (non-Javadoc)
     * @see com.atlassian.integrationtesting.UiTester#currentPage()
     */
    public Doj currentPage()
    {
        return Doj.on(currentPage);
    }
    
    /* (non-Javadoc)
     * @see com.atlassian.integrationtesting.UiTester#getHtmlContents()
     */
    public String getHtmlContents()
    {
        return currentPage().getElement(0).asXml();
    }

    /* (non-Javadoc)
     * @see com.atlassian.integrationtesting.UiTester#elementById(java.lang.String)
     */
    public Doj elementById(String id)
    {
        return currentPage().get("#" + id);
    }

    /* (non-Javadoc)
     * @see com.atlassian.integrationtesting.UiTester#closeWindows()
     */
    public void closeWindows()
    {
        webClient.closeAllWindows();
    }

    public void restore(String backup)
    {
        restore(backup, "admin");
    }

    public void restore(String backup, String username)
    {
        URL url = getClass().getClassLoader().getResource(backup);
        if (url == null)
        {
            throw new RestoreFromBackupException("Backup file " + backup + " not found");
        }
        restore.apply(new Backup(this, url, username));
    }

    public boolean isJavaScriptEnabled()
    {
        return webClient.isJavaScriptEnabled();
    }

    public void setJavaScriptEnabled(boolean enabled)
    {
        webClient.setJavaScriptEnabled(enabled);
    }
}
