package com.atlassian.integrationtesting.runner;

import java.lang.annotation.Annotation;

import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;

import org.junit.internal.runners.model.EachTestNotifier;
import org.junit.runner.Description;
import org.junit.runner.Runner;
import org.junit.runner.notification.RunNotifier;
import org.junit.runners.BlockJUnit4ClassRunner;
import org.junit.runners.model.FrameworkMethod;
import org.junit.runners.model.InitializationError;
import org.junit.runners.model.Statement;
import org.junit.runners.model.TestClass;

import static com.google.common.base.Predicates.and;

/**
 * <p>An easily extendable JUnit {@link Runner}.  Instead of having to try and deal with complex inheritance hierarchies
 * to get the behavior you want, just mix in the behavior you want by subclassing {@code CompositeTestRunner}.</p>
 * 
 * <p>Configuration of a {@CompositeTestRunner} is done by passing in an instance of the {@link Composer} class.  To
 * create a {@code Composer) call the {@link #compose()} method.  {@code Composer}s are immutable builder objects, so
 * you can safely reuse a composer as many times as you like.</p>
 * 
 * <h3>Controlling when tests run</h3>
 * <p>Using a {@code Composer} you can tell JUnit which test classes and which individual test methods should be
 * ignored.  This is done by adding a {@link Predicate} to the {@code Composer} instance you are using to configure
 * your runner.  To control which test classes are run or ignored, add a {@code Predicate} using the 
 * {@link Composer#shouldRunTestsInClass(Predicate)} method, which takes a {@link TestClass} as the parameter.
 * To control which individual tests are run or ignored, add a {@code Predicate} using the
 * {@link Composer#shouldRunTestMethod(Predicate)} method, which takes a {@link FrameworkMethod} as the parameter.  An
 * example of when this is useful is if you want to add a custom annotation to your test classes or methods.  To do
 * that, you just create your {@code Predicate}s and configure your {@code CompositeTestRunner} as follows:</p>
 * <pre>
 *   public class MyTestRunner extends CompositeTestRunner {
 *     private static final Predicate<TestClass> classHasAnnotation = ... 
 *     private static final Predicate<FrameworkMethod> methodHasAnnotation = ... 
 *     public MyTestRunner(Class<?> klass) {
 *       super(klass, compose().shouldRunTestsInClass(classHasAnnotation).shouldRunTestMethod(methodHasAnnotation));
 *     }
 *   }
 * </pre>
 * 
 * <p>The {@code Composer} allows you to add as many conditions for test class and methods as you choose.  When
 * determining if a test class or method should be run or ignored, each predicate is and'd together and the test class
 * or method only run if the result is {@code true}, otherwise the test class or method is ignored.</p>
 * 
 * <h3>Doing something before and after tests in a class are run</h3>
 * <p>To configure the runner to do some extra bit of work before and after any tests in a class are run, use the
 * {@link Composer#beforeTestClass(Function)} and {@link Composer#afterTestClass(Function)} methods.  The functions take
 * {@link BeforeTestClass} and {@link AfterTestClass}, respectively, and have a Void return type - ie they are only run
 * for their side effects.  Using these functions allows you to perform some preparation before the tests in the class
 * are run and cleanup afterwards.  This is useful if you have values you want to inject into static class fields, such
 * as mock or objects that are complicated to build and would clutter test code.</p>
 * <pre>
 *   public class MyTestRunner extends CompositeTestRunner {
 *     private static final Function<BeforeTestClass, Void> injectStatics = ... 
 *     private static final Function<AfterTestClass, Void> cleanupStatics = ... 
 *     public MyTestRunner(Class<?> klass) {
 *       super(klass, compose().beforeTestClass(injectStatics).afterTestClass(cleanupStatics));
 *     }
 *   }
 * </pre>
 * 
 * <p>{@code Composer}s can be configured with as many before and after test class functions as are desired and will
 * run in the order in which they are added.</p>
 * 
 * <h3>Doing something before and after each test method is run</h3>
 * <p>To configure some action to be done before and after each test method is run, use the
 * {@link Composer#beforeTestMethod(Function)} and {@link Composer#afterTestMethod(Function)} methods.  These functions
 * take {@link BeforeTestMethod} and {@link AfterTestMethod} as parameters, giving you access to the instance of the
 * test class being run.  This allows you to do further preparation of the test environment before the test is actually
 * run and cleanup afterwards - such as creating instance field objects as mocks.
 * <pre>
 *   public class MyTestRunner extends CompositeTestRunner {
 *     private static final Function<BeforeTestMethod, Void> inject = ... 
 *     private static final Function<AfterTestMethod, Void> cleanup = ... 
 *     public MyTestRunner(Class<?> klass) {
 *       super(klass, compose().beforeTestMethod(inject).afterTest(cleanup));
 *     }
 *   }
 * </pre>
 * 
 * <p>{@code Composer}s can be configured with as many before and after test method functions as are desired and will
 * run in the order in which they are added.</p>
 */
public abstract class CompositeTestRunner extends BlockJUnit4ClassRunner
{
    private final Predicate<TestClass> shouldRunTestsInClass;
    private final Iterable<Function<BeforeTestClass, Void>> beforeTestClass;
    private final Iterable<Function<AfterTestClass, Void>> afterTestClass;
    private final Predicate<FrameworkMethod> shouldRunTestMethod;
    private final Iterable<Function<BeforeTestMethod, Void>> beforeTestMethod;
    private final Iterable<Function<AfterTestMethod, Void>> afterTestMethod;
   
    public CompositeTestRunner(Class<?> klass, Composer composer) throws InitializationError
    {
        super(klass);
        this.shouldRunTestsInClass = and(composer.shouldRunTestsInClass);
        this.shouldRunTestMethod = and(composer.shouldRunTestMethod);
        this.beforeTestClass = composer.beforeTestClass;
        this.afterTestClass = composer.afterTestClass;
        this.beforeTestMethod = composer.beforeTestMethod;
        this.afterTestMethod = composer.afterTestMethod;
    }

    // Thread local to store the notifier passed in the runChild method. This sucks but it's a hell of a lot easier
    // than trying to override the parents runChild method so that a notifier is passed into the methodBlock method.
    private final ThreadLocal<RunNotifier> notifier = new ThreadLocal<RunNotifier>();

    @Override
    public final void run(RunNotifier notifier)
    {
        if (shouldRunTestsInClass.apply(getTestClass()))
        {
            this.notifier.set(notifier);
            try
            {
                super.run(notifier);
            }
            finally
            {
                this.notifier.set(null);
            }
        }
        else
        {
            notifier.fireTestIgnored(getDescription());
        }
    }

    @Override
    protected final void runChild(FrameworkMethod method, RunNotifier notifier)
    {
        EachTestNotifier eachNotifier = makeNotifier(method, notifier);
        if (shouldRunTestMethod.apply(method))
        {
            super.runChild(method, notifier);
        }
        else
        {
            eachNotifier.fireTestIgnored();
        }
    }
    
    @Override
    protected final Statement withBefores(final FrameworkMethod method, final Object target, Statement statement)
    {
        final Statement next = super.withBefores(method, target, statement);
        return new Statement()
        {
            @Override
            public void evaluate() throws Throwable
            {
                applyAll(beforeTestMethod, new BeforeTestMethod(getTestClass(), method, notifier.get(), target));
                next.evaluate();
            }
        };
    }
    
    @Override
    protected final Statement withAfters(final FrameworkMethod method, final Object target, Statement statement)
    {
        final Statement next = super.withAfters(method, target, statement);
        return new Statement()
        {
            @Override
            public void evaluate() throws Throwable
            {
                try
                {
                    next.evaluate();
                }
                finally
                {
                    applyAll(afterTestMethod, new AfterTestMethod(getTestClass(), method, notifier.get(), target));
                }
            }
        };
    }
    
    @Override
    protected final Statement withBeforeClasses(Statement statement)
    {
        final Statement next = super.withBeforeClasses(statement);
        return new Statement()
        {
            @Override
            public void evaluate() throws Throwable
            {
                applyAll(beforeTestClass, new BeforeTestClass(getTestClass(), notifier.get()));
                next.evaluate();
            }
        };
    }
    
    @Override
    protected final Statement withAfterClasses(Statement statement)
    {
        final Statement next = super.withAfterClasses(statement);
        return new Statement()
        {
            @Override
            public void evaluate() throws Throwable
            {
                try
                {
                    next.evaluate();
                }
                finally
                {
                    applyAll(afterTestClass, new AfterTestClass(getTestClass(), notifier.get()));
                }
            }
        };
    }
    
    private <T> void applyAll(Iterable<Function<T, Void>> fs, T t)
    {
        for (Function<T, Void> f : fs)
        {
            f.apply(t);
        }
    }

    private EachTestNotifier makeNotifier(FrameworkMethod method, RunNotifier notifier)
    {
        Description description= describeChild(method);
        return new EachTestNotifier(notifier, description);
    }

    /**
     * Static factory method for creating empty {@code Composer}s.
     * 
     * @return an empty {@code Composer}
     */
    public static Composer compose()
    {
        return new Composer(ImmutableList.<Predicate<TestClass>>of(), 
            ImmutableList.<Function<BeforeTestClass, Void>>of(),
            ImmutableList.<Function<AfterTestClass, Void>>of(),
            ImmutableList.<Predicate<FrameworkMethod>>of(),
            ImmutableList.<Function<BeforeTestMethod, Void>>of(),
            ImmutableList.<Function<AfterTestMethod, Void>>of());
    }

    /**
     * Allows for easy composition of {@code CompositeTestRunner} instances.  {@code Composer}s are immutable, making
     * them safe to be used from multiple threads and/or to be reused to build multiple runners. 
     */
    public static final class Composer
    {
        private final Iterable<Predicate<TestClass>> shouldRunTestsInClass;
        private final Iterable<Function<BeforeTestClass, Void>> beforeTestClass;
        private final Iterable<Function<AfterTestClass, Void>> afterTestClass;
        private final Iterable<Predicate<FrameworkMethod>> shouldRunTestMethod;
        private final Iterable<Function<BeforeTestMethod, Void>> beforeTestMethod;
        private final Iterable<Function<AfterTestMethod, Void>> afterTestMethod;

        private Composer(Iterable<Predicate<TestClass>> shouldRunTestsInClass,
            Iterable<Function<BeforeTestClass, Void>> beforeTestClass,
            Iterable<Function<AfterTestClass, Void>> afterTestClass,
            Iterable<Predicate<FrameworkMethod>> shouldRunTestMethod,
            Iterable<Function<BeforeTestMethod, Void>> beforeTestMethod,
            Iterable<Function<AfterTestMethod, Void>> afterTestMethod)
        {
            this.shouldRunTestsInClass = shouldRunTestsInClass;
            this.beforeTestClass = beforeTestClass;
            this.afterTestClass = afterTestClass;
            this.shouldRunTestMethod = shouldRunTestMethod;
            this.beforeTestMethod = beforeTestMethod;
            this.afterTestMethod = afterTestMethod;
        }
        
        /**
         * Creates and returns a new {@code Composer} with the predicate appended to the list of predicates determining
         * when methods in test classes are run or ignored.
         * 
         * @param p predicate to determine if methods in a test class should be run or ignored 
         * @return a new {@code Composer} with the predicate appended
         */
        public Composer shouldRunTestsInClass(Predicate<TestClass> p)
        {
            return new Composer(Iterables.concat(shouldRunTestsInClass, ImmutableList.of(p)), beforeTestClass, afterTestClass, shouldRunTestMethod, beforeTestMethod, afterTestMethod);
        }
        
        /**
         * Creates and returns a new {@code Composer} with the function appended to the list of functions to be run
         * before any tests in a class are run.
         * 
         * @param f function to run before tests in a class are run
         * @return a new {@code Composer} with the function appended
         */
        public Composer beforeTestClass(Function<BeforeTestClass, Void> f)
        {
            return new Composer(shouldRunTestsInClass, Iterables.concat(beforeTestClass, ImmutableList.of(f)), afterTestClass, shouldRunTestMethod, beforeTestMethod, afterTestMethod);
        }
        
        /**
         * Creates and returns a new {@code Composer} with the function appended to the list of functions to be run
         * after any tests in a class are run.
         * 
         * @param f function to run after tests in a class are run
         * @return a new {@code Composer} with the function appended
         */
        public Composer afterTestClass(Function<AfterTestClass, Void> f)
        {
            return new Composer(shouldRunTestsInClass, beforeTestClass, Iterables.concat(afterTestClass, ImmutableList.of(f)), shouldRunTestMethod, beforeTestMethod, afterTestMethod);
        }
        
        /**
         * Creates and returns a new {@code Composer} with the predicate appended to the list of predicates determining
         * when individual test methods are run or ignored.
         * 
         * @param p predicate to determine if a test method should be run or ignored 
         * @return a new {@code Composer} with the predicate appended
         */
        public Composer shouldRunTestMethod(Predicate<FrameworkMethod> p)
        {
            return new Composer(shouldRunTestsInClass, beforeTestClass, afterTestClass, Iterables.concat(shouldRunTestMethod, ImmutableList.of(p)), beforeTestMethod, afterTestMethod);
        }
        
        /**
         * Creates and returns a new {@code Composer} with the function appended to the list of functions to be run
         * before individual test methods are run.
         * 
         * @param f function to run before test methods run
         * @return a new {@code Composer} with the function appended
         */
        public Composer beforeTestMethod(Function<BeforeTestMethod, Void> f)
        {
            return new Composer(shouldRunTestsInClass, beforeTestClass, afterTestClass, shouldRunTestMethod, Iterables.concat(beforeTestMethod, ImmutableList.of(f)), afterTestMethod);
        }
        
        /**
         * Creates and returns a new {@code Composer} with the function appended to the list of functions to be run
         * after individual test methods are run.
         * 
         * @param f function to run after test methods run
         * @return a new {@code Composer} with the function appended
         */
        public Composer afterTestMethod(Function<AfterTestMethod, Void> f)
        {
            return new Composer(shouldRunTestsInClass, beforeTestClass, afterTestClass, shouldRunTestMethod, beforeTestMethod, Iterables.concat(afterTestMethod, ImmutableList.of(f)));
        }

        /**
         * Creates and returns a new {@code Composer} with the predicates for test classes and methods AND'd and the
         * functions to be run before and after test classes and methods concatenated such that the functions from
         * {@code this} {@code Composer} will be run first and the function from {@code composer} will be run last.
         * 
         * @param composer composer whose predicates and functions are to be combined with the predicates and functions of {@code this}
         * @return new {@code Composer} combining the predicates and functions from {@code this} and {@code composer}
         */
        public Composer from(Composer composer)
        {
            return new Composer(Iterables.concat(shouldRunTestsInClass, composer.shouldRunTestsInClass),
                Iterables.concat(beforeTestClass, composer.beforeTestClass),
                Iterables.concat(afterTestClass, composer.afterTestClass),
                Iterables.concat(shouldRunTestMethod, composer.shouldRunTestMethod),
                Iterables.concat(beforeTestMethod, composer.beforeTestMethod),
                Iterables.concat(afterTestMethod, composer.afterTestMethod));
        }
    }
    
    /**
     * Parameter type passed to functions that are before all tests in a class.  Includes the {@code TestClass} for
     * reflection and the {@link RunNotifier} that will be used when the tests are run. 
     */
    public static final class BeforeTestClass
    {
        public final TestClass testClass;
        public final RunNotifier notifier;
        
        public BeforeTestClass(TestClass testClass, RunNotifier notifier)
        {
            this.testClass = testClass;
            this.notifier = notifier;
        }

        public boolean hasAnnotation(Class<? extends Annotation> cls)
        {
            return testClass.getJavaClass().isAnnotationPresent(cls);
        }

        public <T extends Annotation> T getAnnotation(Class<T> cls)
        {
            return testClass.getJavaClass().getAnnotation(cls);
        }
    }

    /**
     * Parameter type passed to functions that are after all tests in a class.  Includes the {@link TestClass} for
     * reflection and the {@link RunNotifier} that will be used for reporting when the tests are run. 
     */
    public static final class AfterTestClass
    {
        public final TestClass testClass;
        public final RunNotifier notifier;
        
        public AfterTestClass(TestClass testClass, RunNotifier notifier)
        {
            this.testClass = testClass;
            this.notifier = notifier;
        }
    }
    
    /**
     * Parameter type passed to functions that are before individual test methods.  Includes the {@link TestClass} and
     * {@link FrameworkMethod} for reflection, the {@link RunNotifier} that will be used for reporting when the tests
     * are run, and the actual test class instance. 
     */
    public static final class BeforeTestMethod
    {
        public final TestClass testClass;
        public final FrameworkMethod method;
        public final RunNotifier notifier;
        public final Object target;
        
        private BeforeTestMethod(TestClass testClass, FrameworkMethod method, RunNotifier notifier, Object target)
        {
            this.testClass = testClass;
            this.method = method;
            this.notifier = notifier;
            this.target = target;
        }

        public boolean hasAnnotation(Class<? extends Annotation> cls)
        {
            return method.getAnnotation(cls) != null ||
                method.getMethod().getDeclaringClass().isAnnotationPresent(cls);
        }

        public <T extends Annotation> T getAnnotation(Class<T> cls)
        {
            if (method.getAnnotation(cls) != null)
            {
                return method.getAnnotation(cls);
            }
            else
            {
                return method.getMethod().getDeclaringClass().getAnnotation(cls);
            }
        }
    }

    /**
     * Parameter type passed to functions that are after individual test methods.  Includes the {@link TestClass} and
     * {@link FrameworkMethod} for reflection, the {@link RunNotifier} that will be used for reporting when the tests
     * are run, and the actual test class instance. 
     */
    public static final class AfterTestMethod
    {
        public final TestClass testClass;
        public final FrameworkMethod method;
        public final RunNotifier notifier;
        public final Object target;
        
        private AfterTestMethod(TestClass testClass, FrameworkMethod method, RunNotifier notifier, Object target)
        {
            this.testClass = testClass;
            this.method = method;
            this.notifier = notifier;
            this.target = target;
        }

        public boolean hasAnnotation(Class<? extends Annotation> cls)
        {
            return method.getAnnotation(cls) != null ||
                method.getMethod().getDeclaringClass().isAnnotationPresent(cls);
        }

        public <T extends Annotation> T getAnnotation(Class<T> cls)
        {
            if (method.getAnnotation(cls) != null)
            {
                return method.getAnnotation(cls);
            }
            else
            {
                return method.getMethod().getDeclaringClass().getAnnotation(cls);
            }
        }
    }
}
