package com.atlassian.integrationtesting.runner;

import java.io.IOException;
import java.io.UnsupportedEncodingException;

import com.atlassian.integrationtesting.ApplicationPropertiesProvidingModule;
import com.atlassian.integrationtesting.runner.CompositeTestRunner.AfterTestClass;
import com.atlassian.integrationtesting.runner.CompositeTestRunner.BeforeTestClass;
import com.atlassian.integrationtesting.runner.CompositeTestRunner.Composer;
import com.atlassian.sal.api.ApplicationProperties;

import com.google.common.base.Function;
import com.google.inject.AbstractModule;
import com.google.inject.Guice;
import com.google.inject.Inject;
import com.google.inject.Injector;
import com.google.inject.Scopes;

import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.NameValuePair;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.HttpClient;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.http.message.BasicNameValuePair;
import org.junit.runner.Description;
import org.junit.runner.notification.RunListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static java.util.Arrays.asList;

/**
 * This is a run listener that communicates with application when test is starting and when it is finished.  It assumes
 * that the ait-plugin is installed and the ${baseurl}/rest/ait/1.0/test-log resource is available. 
 */
public final class ApplicationLogDelimiter extends RunListener
{
    private static final Logger logger = LoggerFactory.getLogger(ApplicationLogDelimiter.class);
    
    private final ApplicationProperties applicationProperties;
    private final HttpClient client = new DefaultHttpClient();

    @Inject
    ApplicationLogDelimiter(ApplicationProperties applicationProperties)
    {
        this.applicationProperties = applicationProperties;
    }
    
    /**
     * Creates a {@code Composer} configured with functions to add a {@code ApplicationLogDelimiter} before tests in 
     * a class are run and remove the listener when the tests in the class are complete.
     * 
     * @param applicationProperties Required to get the application base url to use when logging tests status
     * @return composer to add and remove {@code ApplicationLogDelimiter} instance during test runs 
     */
    public static Composer compose(ApplicationProperties applicationProperties)
    {
        Injector injector = Guice.createInjector(new ApplicationPropertiesProvidingModule(applicationProperties));
        return compose(injector);
    }
    
    /**
     * Creates a {@code Composer} using the given injector.  The injector is expected to have been configured with a
     * binding for an {@link ApplicationProperties} instance.  If it hasn't been properly configured, a configuration
     * exception will be thrown.
     * 
     * @param injector Injector providing a binding for {@code ApplicationProperties}
     * @return composer to add and remove {@code ApplicationLogDelimiter} instance during test runs
     */
    public static Composer compose(Injector injector)
    {
        Injector child = injector.createChildInjector(new LogDelimiterModule());
        return CompositeTestRunner.compose().
                beforeTestClass(new LogDelimiterBefore(child)).afterTestClass(new LogDelimiterAfter(child));
    }
    
    @Override
    public void testStarted(Description description) throws Exception
    {
        post(State.TEST_STARTED, description.getDisplayName());
    }

    @Override
    public void testFinished(Description description) throws Exception
    {
        post(State.TEST_FINISHED, description.getDisplayName());
    }
    
    private void post(State state, String name)
    {
        HttpPost post = new HttpPost(applicationProperties.getBaseUrl() + "/rest/ait/1.0/test-log");
        post.setEntity(form(pair("state", state.toString()), pair("test", name)));
        try
        {
            HttpResponse response = client.execute(post);
            if (response.getEntity() != null)
            {
                // if there is a body, consume it to release the connection 
                response.getEntity().consumeContent();
            }
        }
        catch (ClientProtocolException e)
        {
            logger.error("Could not log test state", e);
        }
        catch (IOException e)
        {
            logger.error("Could not log test state", e);
        }
    }
    
    private static final HttpEntity form(NameValuePair... parameters)
    {
        try
        {
            return new UrlEncodedFormEntity(asList(parameters));
        }
        catch (UnsupportedEncodingException e)
        {
            // really shouldn't happen, the default encoding should be available on all JVMs
            throw new RuntimeException(e);
        }
    }
    
    private static final NameValuePair pair(String name, String value)
    {
        return new BasicNameValuePair(name, value);
    }
    
    enum State
    {
        TEST_STARTED, TEST_FINISHED
    }
    
    private static final class LogDelimiterModule extends AbstractModule
    {
        @Override
        protected void configure()
        {
            bind(ApplicationLogDelimiter.class).in(Scopes.SINGLETON);
        }
    }

    private static final class LogDelimiterBefore implements Function<BeforeTestClass, Void>
    {
        private final Injector injector;

        public LogDelimiterBefore(Injector injector)
        {
            this.injector = injector;
        }
        
        public Void apply(BeforeTestClass test)
        {
            test.notifier.addListener(injector.getInstance(ApplicationLogDelimiter.class));
            return null;
        }
    }
    
    private static final class LogDelimiterAfter implements Function<AfterTestClass, Void>
    {
        private final Injector injector;

        public LogDelimiterAfter(Injector injector)
        {
            this.injector = injector;
        }

        public Void apply(AfterTestClass test)
        {
            test.notifier.removeListener(injector.getInstance(ApplicationLogDelimiter.class));
            return null;
        }
    }
}
