package com.atlassian.extras.decoder.v2;

import com.atlassian.extras.common.LicenseException;
import com.atlassian.extras.common.org.springframework.util.DefaultPropertiesPersister;
import com.atlassian.extras.decoder.api.AbstractLicenseDecoder;
import com.atlassian.extras.decoder.api.LicenseVerificationException;
import com.atlassian.extras.keymanager.KeyManager;
import com.atlassian.extras.keymanager.SortedProperties;
import org.apache.commons.codec.binary.Base64;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.StringWriter;
import java.io.UnsupportedEncodingException;
import java.nio.charset.StandardCharsets;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Properties;
import java.util.zip.Inflater;
import java.util.zip.InflaterInputStream;

import static com.atlassian.extras.common.LicensePropertiesConstants.CREATION_DATE;
import static com.atlassian.extras.common.LicensePropertiesConstants.LICENSE_HASH;
import static com.atlassian.extras.common.LicensePropertiesConstants.LICENSE_HASH_KEY_VERSION;
import static com.atlassian.extras.decoder.api.LicenseVerificationException.VerificationFailureReason.ERROR_DURING_VERIFICATION;
import static com.atlassian.extras.decoder.api.LicenseVerificationException.VerificationFailureReason.MISSING_PROPERTY;
import static com.atlassian.extras.decoder.api.LicenseVerificationException.VerificationFailureReason.VERIFICATION_FAILED;
import static com.atlassian.extras.keymanager.PublicKeys.LICENSE_STRING_KEY_V2_VERSION;


/**
 * This decoder expects the following format:
 * <table>
 *     <caption>License format</caption>
 *     <thead>
 *         <tr>
 *             <th>License text length</th>
 *             <th>License text</th>
 *             <th>License text md5 hash</th>
 *             <th>Separator</th>
 *             <th>License version</th>
 *             <th>Length of base 64 encoded part</th>
 *         </tr>
 *     </thead>
 *     <tbody>
 *         <tr>
 *             <td colspan="3">Base 64 encoded</td>
 *             <td>X</td>
 *             <td>Clear text</td>
 *             <td>Base 31 encoded</td>
 *         </tr>
 *     </tbody>
 * </table>
 */
public class Version2LicenseDecoder extends AbstractLicenseDecoder {
    
    public static final int VERSION_NUMBER_1 = 1;
    public static final int VERSION_NUMBER_2 = 2;

    public static final int VERSION_LENGTH = 3;
    public static final int ENCODED_LICENSE_LENGTH_BASE = 31;
    public static final byte[] LICENSE_PREFIX = new byte[]{0xD, 0xE, 0xC, 0xA, 0xF};

    public static final char SEPARATOR = 'X';
    private static final int ENCODED_LICENSE_LINE_LENGTH = 76;

    private static final Date LICENSE_HASH_CUTOFF_DATE = new Date(2020 - 1900, 12, 1);

    private volatile boolean verifyLicenseHash = false;
    private volatile boolean skipVerificationBeforeCutoffDate = false;

    public Version2LicenseDecoder() {
    }

    public Version2LicenseDecoder(boolean verifyLicenseHash, boolean skipVerificationBeforeCutoffDate) {
        this.verifyLicenseHash = verifyLicenseHash;
        this.skipVerificationBeforeCutoffDate = skipVerificationBeforeCutoffDate;
    }

    public boolean canDecode(String licenseString) {
        licenseString = removeWhiteSpaces(licenseString);

        final int pos = licenseString.lastIndexOf(SEPARATOR);
        if (pos == -1 || pos + VERSION_LENGTH >= licenseString.length()) {
            return false;
        }

        try {
            // verify the version number
            final int version = Integer.parseInt(licenseString.substring(pos + 1, pos + VERSION_LENGTH));
            if (version != VERSION_NUMBER_1 && version != VERSION_NUMBER_2) {
                return false;
            }

            final String lengthStr = licenseString.substring(pos + VERSION_LENGTH);
            int encodedLicenseLength = Integer.valueOf(lengthStr, ENCODED_LICENSE_LENGTH_BASE).intValue();
            if (pos != encodedLicenseLength) {
                return false;
            }

            return true;
        } catch (NumberFormatException e) {
            return false;
        }
    }

    public Properties doDecode(String licenseString) throws LicenseVerificationException {
        final String encodedLicenseTextAndHash = getLicenseContent(removeWhiteSpaces(licenseString));
        final byte[] zippedLicenseBytes = checkAndGetLicenseText(encodedLicenseTextAndHash);
        final Reader licenseText = unzipText(zippedLicenseBytes);
        final Properties properties = loadLicenseConfiguration(licenseText);

        if (verifyLicenseHash) {
            verifyLicenseHash(properties);
        }

        return properties;
    }

    private void verifyLicenseHash(final Properties properties) throws LicenseVerificationException {
        if (skipVerificationBeforeCutoffDate) {
            String creationDate = properties.getProperty(CREATION_DATE);
            if (creationDate == null) {
                throw new LicenseVerificationException(MISSING_PROPERTY, CREATION_DATE, properties);
            }

            try {
                Date created = new SimpleDateFormat("yyyy-MM-dd").parse(creationDate);
                if (created.before(LICENSE_HASH_CUTOFF_DATE)) {
                    return;
                }
            } catch (Exception e) {
                throw new LicenseVerificationException(ERROR_DURING_VERIFICATION, properties, e);
            }
        }

        // Cloning the properties as we need to make modifications to it in order to perform the verification.
        // Using a custom Properties implementation to keep their order consistent
        // when encoding and decoding as this can affect the signature verification.
        SortedProperties clonedProps = new SortedProperties();
        clonedProps.putAll(properties);

        // The licenseHash prop was not part of the license props used to generate the hash.
        // Removing it from the payload before we verify the signature.
        String licenseHash = (String) clonedProps.remove(LICENSE_HASH);
        if (licenseHash == null) {
            throw new LicenseVerificationException(MISSING_PROPERTY, LICENSE_HASH, properties);
        }

        String keyVersion = clonedProps.getProperty(LICENSE_HASH_KEY_VERSION);
        if (keyVersion == null) {
            throw new LicenseVerificationException(MISSING_PROPERTY, LICENSE_HASH_KEY_VERSION, properties);
        }

        boolean verified;

        try {
            StringWriter out = new StringWriter();
            new DefaultPropertiesPersister().store(clonedProps, out, null, true);
            String encodedProps = new String(Base64.encodeBase64(out.toString().getBytes(StandardCharsets.UTF_8)), StandardCharsets.UTF_8);
            verified = KeyManager.getInstance().verify(encodedProps, licenseHash, keyVersion);
        } catch (Exception e) {
            throw new LicenseVerificationException(ERROR_DURING_VERIFICATION, properties, e);
        }

        if (!verified) {
            throw new LicenseVerificationException(VERIFICATION_FAILED, properties);
        }
    }

    protected int getLicenseVersion() {
        return 2;
    }

    private Reader unzipText(byte[] licenseText) {
        final ByteArrayInputStream in = new ByteArrayInputStream(licenseText);
        in.skip(LICENSE_PREFIX.length); // skip the prefix
        final InflaterInputStream zipIn = new InflaterInputStream(in, new Inflater());
        return new InputStreamReader(zipIn, StandardCharsets.UTF_8);
    }

    private String getLicenseContent(String licenseString) {
        final String lengthStr = licenseString.substring(licenseString.lastIndexOf(SEPARATOR) + VERSION_LENGTH);
        try {
            int encodedLicenseLength = Integer.valueOf(lengthStr, ENCODED_LICENSE_LENGTH_BASE).intValue();
            return licenseString.substring(0, encodedLicenseLength);
        } catch (NumberFormatException e) {
            throw new LicenseException("Could NOT decode license length <" + lengthStr + ">", e);
        }
    }

    private byte[] checkAndGetLicenseText(String licenseContent) {
        final byte[] licenseText;
        try {
            byte decodedBytes[] = Base64.decodeBase64(licenseContent.getBytes(StandardCharsets.UTF_8));
            ByteArrayInputStream in = new ByteArrayInputStream(decodedBytes);
            DataInputStream dIn = new DataInputStream(in);
            int textLength = dIn.readInt();
            licenseText = new byte[textLength];
            dIn.read(licenseText);
            byte[] hash = new byte[dIn.available()];
            dIn.read(hash);

            String encodedLicenseText = new String(Base64.encodeBase64(licenseText), StandardCharsets.UTF_8);
            String encodedHash = new String(Base64.encodeBase64(hash), StandardCharsets.UTF_8);

            if (!KeyManager.getInstance().verify(encodedLicenseText, encodedHash, LICENSE_STRING_KEY_V2_VERSION)) {
                throw new LicenseException("Failed to verify the license.");
            }
        } catch (Exception e) {
            // should never happen!
            throw new LicenseException(e);
        }

        return licenseText;
    }

    private Properties loadLicenseConfiguration(Reader text) {
        try {
            Properties props = new Properties();
            new DefaultPropertiesPersister().load(props, text);
            return props;
        } catch (IOException e) {
            throw new LicenseException("Could NOT load properties from reader", e);
        }
    }

    /**
     * Removes all the whitespace characters. Whitespace is defined by {@link Character#isWhitespace}.
     */
    private static String removeWhiteSpaces(String licenseData) {
        if (licenseData == null || licenseData.length() == 0) {
            return licenseData;
        }

        char[] chars = licenseData.toCharArray();
        StringBuffer buf = new StringBuffer(chars.length);
        for (int i = 0; i < chars.length; i++) {
            if (!Character.isWhitespace(chars[i])) {
                buf.append(chars[i]);
            }
        }

        return buf.toString();
    }

    /*
     * This method has to be public for the sake of the legacy code for the transition period.
     * It is used in LicensePair for handling the new license for the old code base.
     */
    public static String packLicense(byte[] text, byte[] hash) throws LicenseException {
        try {
            // start over - we will write text length + text + hash
            ByteArrayOutputStream out = new ByteArrayOutputStream();
            DataOutputStream dOut = new DataOutputStream(out);
            dOut.writeInt(text.length);
            dOut.write(text);
            dOut.write(hash);

            byte[] allData = out.toByteArray();
            String result = new String(Base64.encodeBase64(allData), StandardCharsets.UTF_8).trim(); // split the license data into chunks and remove potential new line at the end

            // for verification we will append a separator + license version (0-padded - 2 characters long) + encoded license length
            // this will allow us to decide which decoder to use and which version of the new decoder to use
            // the length is encoded using radix 31 to make sure our separator never shows up in the encoded license length
            result = result + SEPARATOR + "0" + VERSION_NUMBER_2 + Integer.toString(result.length(), ENCODED_LICENSE_LENGTH_BASE);
            result = split(result);
            return result;
        } catch (IOException e) {
            // should never happen!
            throw new LicenseException(e);
        }
    }

    /**
     * Splits the license into lines of maximum 76 character long and inserts '\n' character between the lines
     */
    private static String split(String licenseData) {
        if (licenseData == null || licenseData.length() == 0) {
            return licenseData;
        }

        char[] chars = licenseData.toCharArray();
        StringBuffer buf = new StringBuffer(chars.length + (chars.length / ENCODED_LICENSE_LINE_LENGTH));
        for (int i = 0; i < chars.length; i++) {
            buf.append(chars[i]);
            if (i > 0 && i % ENCODED_LICENSE_LINE_LENGTH == 0) {
                buf.append('\n');
            }
        }

        return buf.toString();
    }

    public void setVerifyLicenseHash(boolean verifyLicenseHash) {
        this.verifyLicenseHash = verifyLicenseHash;
    }

    public void setSkipVerificationBeforeCutoffDate(boolean skipVerificationBeforeCutoffDate) {
        this.skipVerificationBeforeCutoffDate = skipVerificationBeforeCutoffDate;
    }
}
