package com.atlassian.diagnostics;

import com.google.common.base.MoreObjects;
import com.google.common.base.Objects;
import org.apache.commons.lang3.StringUtils;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import static com.atlassian.diagnostics.DiagnosticsConstants.PLUGIN_NOT_DETECTED_KEY;
import static com.google.common.base.MoreObjects.firstNonNull;

public class PluginDetails {

    private final String key;
    private final String name;
    private final String version;

    public PluginDetails(String key, String name, String version) {
        this.key = MoreObjects.firstNonNull(StringUtils.trimToNull(key), PLUGIN_NOT_DETECTED_KEY);
        this.name = firstNonNull(name, key);
        this.version = version;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        PluginDetails plugin = (PluginDetails) o;
        return Objects.equal(key, plugin.key) &&
                Objects.equal(version, plugin.version);
    }

    /**
     * @return the plugin key, or {@link DiagnosticsConstants#PLUGIN_NOT_DETECTED_KEY} if the
     *         plugin was not detected
     */
    @Nonnull
    public String getKey() {
        return key;
    }

    /**
     * @return the full name of the plugin
     */
    @Nonnull
    public String getName() {
        return name;
    }

    /**
     * @return the plugin version, if available
     */
    @Nullable
    public String getVersion() {
        return version;
    }

    @Override
    public int hashCode() {
        return Objects.hashCode(key, version);
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper("")
                .addValue(key)
                .addValue(name)
                .addValue(version)
                .toString();
    }
}
