package com.atlassian.diagnostics;

import javax.annotation.Nonnull;

/**
 * A callback for iterating through a stream of items. After an initial {@link #onStart(PageRequest) call} to
 * signify zero or more items are to follow, the callback is repeatedly {@link #onItem(T) called with an item}
 * until the last one is processed in the stream or until {@link CallbackResult#DONE} is returned by
 * {@link #onItem(T)}. At the end of processing, {@link #onEnd(PageSummary)} is called and any value it
 * returned deemed, by the enclosing call or method, as the result of the stream.
 *
 * @param <R> the type of the possibly null value to be returned from {@link #onEnd(PageSummary)}
 */
@FunctionalInterface
public interface PageCallback<T, R> {
    /**
     * Called before the first item is encountered. Note that this will be called even if no item is passed to this
     * callback (for instance if any associated criteria with the streaming operation failed to find matching items).
     */
    default void onStart(@Nonnull PageRequest pageRequest) {
    }

    /**
     * Called exactly once for each item that matches any criteria or other qualifier of the enclosing call or process
     * @param item the current item
     * @return {@link CallbackResult#CONTINUE} if the next item should be processed or {@link CallbackResult#DONE}
     *         to indicate premature termination of the streaming
     * operation
     */
    @Nonnull
    CallbackResult onItem(@Nonnull T item);

    /**
     * Called after the last item has been passed to this callback (or if no item are to be passed then immediately
     * after {@link #onStart(PageRequest)}.
     *
     * @return any result the callback wishes the enclosing call or process to return to its caller or {@code null}
     * if no such result is to be supplied (or if the result is in fact {@code null})
     */
    default R onEnd(@Nonnull PageSummary summary) {
        return null;
    }
}
