package com.atlassian.db.config.password;

import com.atlassian.db.config.password.ciphers.base64.Base64Cipher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Optional;

public class DefaultCipherProvider implements CipherProvider {

    private final static Logger log = LoggerFactory.getLogger(DefaultCipherProvider.class);

    /**
     * @return {@link Base64Cipher}
     */
    @Override
    public String getDefaultCipherProviderClassName() {
        return Base64Cipher.class.getCanonicalName();
    }

    /**
     * Uses reflection to obtain an instance of the {@link Cipher} implementation.
     *
     * @param cipherProviderClassName canonical name of the {@link Class} which implements {@link Cipher} and should be initiated.
     * @return new instance of {@link Cipher} packed inside Optional or {@link Optional#empty()} in case passed param is null or empty.
     * @throws IllegalArgumentException in case of {@link ClassNotFoundException}, {@link InstantiationException} or {@link IllegalAccessException}
     */
    @Override
    public Optional<Cipher> getInstance(final String cipherProviderClassName) {
        log.debug("Initiate cipher provider class: " + cipherProviderClassName);
        if (cipherProviderClassName == null || cipherProviderClassName.isEmpty()) {
            return Optional.empty();
        }
        try {
            final Class<?> cipherProviderClass = Class.forName(cipherProviderClassName);
            final Cipher provider = (Cipher) cipherProviderClass.newInstance();
            return Optional.of(provider);
        } catch (final ClassNotFoundException | InstantiationException | IllegalAccessException classInitException) {
            log.error("Couldn't initiate class: " + cipherProviderClassName, classInitException);
            throw new IllegalArgumentException(classInitException);
        }
    }
}
