package com.atlassian.crowd.manager.validation;

import com.atlassian.crowd.manager.proxy.TrustedProxyManager;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;

import javax.servlet.http.HttpServletRequest;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.Arrays;
import java.util.List;

/**
 * Utility class for handling the X-Forwarded-For (XFF) HTTP request header.
 *
 * @since 2.2
 */
public class XForwardedForUtil {
    private static final String X_FORWARDED_FOR = "X-Forwarded-For";

    private XForwardedForUtil() {
    }

    /**
     * Returns the originating client address the proxies are forwarding for if the proxies are trusted, otherwise, return the
     * request address.
     *
     * @param trustedProxyManager used to determine if the proxy address is trusted
     * @param request             HTTP request
     * @return originating client address if the proxies are trusted, otherwise, the request address is returned
     */
    public static InetAddress getTrustedAddress(TrustedProxyManager trustedProxyManager, HttpServletRequest request) {
        final String trustedAddress = getTrustedAddress(trustedProxyManager, request.getRemoteAddr(), request.getHeader(X_FORWARDED_FOR));
        try {
            // Use InetAddress as Guava cannot currently parse IPv6 scopes
            return InetAddress.getByName(trustedAddress);
        } catch (UnknownHostException e) {
            throw new RuntimeException(e); // This should never happen
        }
    }

    /**
     * Returns the originating client address the proxies are forwarding for if all the proxies are trusted, otherwise,
     * return the request address.
     *
     * @param trustedProxyManager used to determine if the proxy address is trusted
     * @param requestAddress      HTTP request address
     * @param xForwardedFor       X-Forwarded-For header
     * @return originating client address if the proxies are trusted, otherwise, the request address is returned
     */
    public static String getTrustedAddress(TrustedProxyManager trustedProxyManager, String requestAddress, String xForwardedFor) {
        List<String> chain = Lists.newArrayList(StringUtils.split(Strings.nullToEmpty(xForwardedFor), ", "));
        chain.add(requestAddress);
        return (chain.subList(1, chain.size()).stream().allMatch(trustedProxyManager::isTrusted)) ?
                chain.get(0) : requestAddress;
    }
}
