package com.atlassian.crowd.manager.authentication;

import com.atlassian.crowd.exception.ApplicationNotFoundException;
import com.atlassian.crowd.exception.DirectoryNotFoundException;
import com.atlassian.crowd.exception.ExpiredCredentialException;
import com.atlassian.crowd.exception.InactiveAccountException;
import com.atlassian.crowd.exception.InvalidAuthenticationException;
import com.atlassian.crowd.exception.InvalidTokenException;
import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.crowd.exception.UserNotFoundException;
import com.atlassian.crowd.manager.application.ApplicationAccessDeniedException;
import com.atlassian.crowd.model.application.Application;
import com.atlassian.crowd.model.authentication.ApplicationAuthenticationContext;
import com.atlassian.crowd.model.authentication.UserAuthenticationContext;
import com.atlassian.crowd.model.authentication.ValidationFactor;
import com.atlassian.crowd.model.token.Token;
import com.atlassian.crowd.model.token.TokenLifetime;
import com.atlassian.crowd.model.user.User;

import javax.annotation.Nullable;
import java.util.Date;
import java.util.List;
import java.util.Optional;

public interface TokenAuthenticationManager {
    /**
     * Authenticates an application and generates an authentication token.
     *
     * @param application           the application being authenticated
     * @param authenticationContext application authentication credentials.
     * @param tokenLifetime         Requested lifetime of the token
     * @return generated authentication token.
     * @throws com.atlassian.crowd.exception.InvalidAuthenticationException authentication was not successful because either the application does not exist, the password is incorrect, the application is inactive or there was a problem generating the authentication token.
     */
    Token authenticateApplication(Application application, ApplicationAuthenticationContext authenticationContext, TokenLifetime tokenLifetime) throws InvalidAuthenticationException;

    /**
     * Authenticates an application and generates an authentication token, ignoring the credentials.
     * <p>
     * This method should only be used to generate a token for an application that has already authenticated via
     * some other means (eg. TLS client certificates) as this method bypasses any password checks.
     *
     * @param application           the application being authenticated
     * @param authenticationContext application authentication credentials.
     * @param tokenLifetime         Requested lifetime of the token
     * @return generated authentication token.
     * @throws com.atlassian.crowd.exception.InvalidAuthenticationException authentication was not successful because either the application does not exist, the application is inactive or there was a problem generating the authentication token.
     * @since 2.9
     */
    Token authenticateApplicationWithoutValidatingPassword(Application application, ApplicationAuthenticationContext authenticationContext, TokenLifetime tokenLifetime) throws InvalidAuthenticationException;

    /**
     * Authenticates a user and and generates an authentication token. The password of the user is validated before generating a token.
     * <p>
     * The {@link com.atlassian.crowd.directory.RemoteDirectory#authenticate(String, com.atlassian.crowd.embedded.api.PasswordCredential)} method is
     * iteratively called for each assigned directory. If the user does not exist in one directory, the directory is skipped and the next one is examined. If the user does
     * not exist in any of the assigned directories then an {@link com.atlassian.crowd.exception.InvalidAuthenticationException} is thrown.
     *
     *
     * @param application
     * @param authenticateContext The authentication details for the user.
     * @param tokenLifetime       Requested lifetime of the token
     * @return The authenticated token for the user.
     * @throws InvalidAuthenticationException   The authentication was not successful.
     * @throws OperationFailedException         error thrown by directory implementation when attempting to find or authenticate the user.
     * @throws InactiveAccountException         user account is inactive.
     * @throws ApplicationAccessDeniedException user does not have access to authenticate with application.
     * @throws ExpiredCredentialException       the user's credentials have expired. The user must change their credentials in order to successfully authenticate.
     */
    Token authenticateUser(Application application, UserAuthenticationContext authenticateContext, TokenLifetime tokenLifetime)
            throws InvalidAuthenticationException, OperationFailedException, InactiveAccountException, ApplicationAccessDeniedException, ExpiredCredentialException;

    /**
     * Feigns the authentication process for a user and creates a token for the authentication <b>without</b> validating the password.
     * <p>
     * This method should only be used to generate a token for a user that has already authenticated credentials via
     * some other means (eg. SharePoint NTLM connector) as this method bypasses any password checks.
     * <p>
     * If you want actual password authentication, use the {@link #authenticateUser(Application, UserAuthenticationContext, TokenLifetime)} method.
     *
     *
     * @param application
     * @param authenticateContext The authentication details for the user.
     * @return The authenticated token for the user.
     * @throws InvalidAuthenticationException   if the authentication was not successful.
     * @throws OperationFailedException         if the error thrown by directory implementation when attempting to find or authenticate the user.
     * @throws InactiveAccountException         if the user account is inactive.
     * @throws ApplicationAccessDeniedException if the user does not have access to authenticate with application.
     */
    Token authenticateUserWithoutValidatingPassword(Application application, UserAuthenticationContext authenticateContext)
            throws InvalidAuthenticationException, OperationFailedException, InactiveAccountException, ApplicationAccessDeniedException;

    /**
     * Validates an application token key given validation factors.
     *
     * @param tokenKey          returns a valid token corresponding to the tokenKey.
     * @param validationFactors validation factors for generating the token hash.
     * @return validated token.
     * @throws InvalidTokenException if the tokenKey or corresponding client validation factors do not represent a valid application token.
     */
    Token validateApplicationToken(String tokenKey, ValidationFactor[] validationFactors) throws InvalidTokenException;

    /**
     * Validates a user token key given validation factors and checks that the user is allowed to authenticate
     * with the specified application
     *
     * @param application       the application performing the authentication
     * @param userTokenKey      returns a valid token corresponding to the tokenKey.
     * @param validationFactors validation factors for generating the token hash.
     * @return validated authentication token.
     * @throws InvalidTokenException                                  if the userTokenKey or corresponding validationFactors do not represent a valid SSO token.
     * @throws com.atlassian.crowd.exception.OperationFailedException there was an error communicating with an underlying directory when determining if a user is allowed to authenticate with the application (eg. if a user has the appropriate group memberships).
     * @throws ApplicationAccessDeniedException                       the user is not allowed to authenticate with the application.
     */
    Token validateUserToken(Application application, String userTokenKey, ValidationFactor[] validationFactors)
            throws InvalidTokenException, ApplicationAccessDeniedException, OperationFailedException;


    /**
     * Attempts to invalidate a Token based on the passed in Token key (random hash).
     * <p>
     * If the token does not exist (ie. already invalidated) this method returns {@link Optional#empty()}. If an existing token is successfully invalidated, a
     * TokenInvalidatedEvent is fired, and the invalidated token is returned
     *
     * @param token the token key (random hash) to invalidate.
     */
    Optional<Token> invalidateToken(String token);

    /**
     * Invalidates all user and application tokens.
     *
     * This means it will also invalidate the token of the calling application.
     */
    void invalidateAllTokens();

    /**
     * Removes all tokens that have exceeded their expiry time.
     * <p>
     * NOTE: Do not call this method from the web layer, as this is wrapped in a Spring managed transaction.
     */
    void removeExpiredTokens();

    /**
     * Will find a user via the passed in token.
     *
     * @param token            the token
     * @param application     the application to do the lookup for
     * @return the User associated to the given token
     * @throws InvalidTokenException            if the User or Directory cannot be found that relates to the given token,
     *                                          or the token is associated to an Application and not a User
     * @throws OperationFailedException         if there was an issue accessing the user from the underlying directory
     */
    User findUserByToken(Token token, Application application)
            throws InvalidTokenException, OperationFailedException;

    /**
     * Returns the token matching a given key
     *
     * @param tokenKey        the token key
     * @param application     the application to do the lookup for
     * @return the Token with the given token key
     * @throws InvalidTokenException            if the token cannot be found by the give key,
     *                                          or the token is associated to an Application and not a User
     * @throws OperationFailedException         if there was an issue accessing the user from the underlying directory
     * @throws ApplicationAccessDeniedException the user is not allowed to authenticate with the application.
     */
    Token findUserTokenByKey(final String tokenKey, Application application)
            throws InvalidTokenException, ApplicationAccessDeniedException, OperationFailedException;

    /**
     * Returns a list of applications a user
     * is authorised to authenticate with.
     * <p>
     * NOTE: this is a potentially expensive call, iterating
     * all applications and all group mappings for
     * each application and determining group membership,
     * ie. expense = number of applications * number of group mappings per application.
     *
     * @param user            user to search for.
     * @param applicationName name of the current application
     * @return list of applications.
     * @throws OperationFailedException     if there was an error querying directory.
     * @throws DirectoryNotFoundException   if the directory could not be found.
     * @throws ApplicationNotFoundException if the application could not be found
     */
    List<Application> findAuthorisedApplications(User user, String applicationName)
            throws OperationFailedException, DirectoryNotFoundException, ApplicationNotFoundException;

    /**
     * Invalidates all sessions for a user, possibly excluding a specific one.
     *
     * @param applicationName name of the current application
     * @param exclusionToken  the random hash of a token to leave valid
     */
    void invalidateTokensForUser(String username, @Nullable String exclusionToken, String applicationName)
            throws UserNotFoundException, ApplicationNotFoundException;

    /**
     * Returns the expiry time of a token.
     *
     * @param token a token
     * @return the expiry time for the given token
     */
    Date getTokenExpiryTime(Token token);
}
