package com.atlassian.crowd.licensing;

import com.atlassian.crowd.exception.ApplicationNotFoundException;
import com.atlassian.crowd.exception.ObjectNotFoundException;
import com.atlassian.crowd.model.ApplicationSubtype;
import com.atlassian.crowd.model.application.Application;
import com.atlassian.crowd.model.licensing.ApplicationLicensedDirectory;
import com.atlassian.crowd.model.licensing.ApplicationLicensedUser;
import com.atlassian.crowd.model.licensing.ApplicationLicensingSummary;
import com.atlassian.fugue.Pair;

import java.util.List;
import java.util.Optional;

/**
 * Component controlling the feature of application license usage monitoring, it is a god class for fetching usage data
 * from the application as well as storing the data and serving the queries for it.
 */
public interface ApplicationLicensingService {

    /**
     * Checks whether license usage data can be fetched/shown for the given application
     */
    boolean canShowLicenseUsageForApplication(Long applicationId) throws ApplicationNotFoundException;

    /**
     * Checks whether requested version is up to date or not for the given {@link ApplicationSubtype} if supplied
     */
    boolean isVersionUptoDate(Long applicationId, String appSubType, Long version) throws ApplicationNotFoundException;

    /**
     * Returns list of jira sub-types available for the given {@link com.atlassian.crowd.model.application.ApplicationType#JIRA}
     * application
     *
     * @throws IllegalArgumentException if application is not Jira type
     */
    List<ApplicationSubtype> listJiraTypes(Long applicationId, Long version) throws ApplicationNotFoundException;

    /**
     * Returns list of directories linked to the users who are consuming licenses in the given application
     *
     * @param applicationId application id to search users for
     * @param appSubType    Jira subtype, only if the application type is Jira
     * @param version       version of data requested
     * @param start         start offset for the results
     * @param limit         maximum number of results
     * @throws ApplicationNotFoundException if application id invalid
     * @throws IllegalArgumentException     If version is not available
     * @throws IllegalStateException        If supplied application sub type is invalid
     */
    List<ApplicationLicensedDirectory> listDirectories(Long applicationId, String appSubType, Long version, int start, int limit)
            throws ApplicationNotFoundException;

    /**
     * Returns list of license consuming users in the products along with the total count of users matching the supplied
     * query
     *
     * @param applicationId    application id
     * @param searchText       search text to be matched across username, email and full name
     * @param directoryId      directory id to which the user belong
     * @param appSubType       application subtype, only applicable for Jira application
     * @param lastLoggedInDate last logged in date on or before
     * @param version          version
     * @param start            offset
     * @param limit            max number of results
     */
    Pair<List<ApplicationLicensedUser>, Long> searchLicensedUsers(Long applicationId, String searchText, String directoryId,
                                                                  String appSubType, Long lastLoggedInDate, Long version, int start, int limit)
            throws ObjectNotFoundException;

    /**
     * Return the licensing summary if available
     */
    Optional<ApplicationLicensingSummary> getLicensingSummary(Long applicationId, String appSubType, Long version)
            throws ApplicationNotFoundException;

    /**
     * Fetches the license usage data from the application and updates the local copy
     *
     * @return true if application data was updated, or else false
     */
    boolean updateApplicationData(Application application);

    /**
     * Schedules updating application license usage data immediately
     */
    void scheduleRefreshApplicationDataJobImmediately(long applicationId) throws ApplicationNotFoundException;

    /**
     * Clears all jobs scheduled for an application
     */
    void clearAllJobs(Application application);

    /**
     * Checks if licensing is configured for specific application
     *
     * @param applicationId ID of application to check
     */
    boolean isLicensingConfigured(Long applicationId) throws ApplicationNotFoundException;
}
