package com.atlassian.crowd.emailchange;

import com.atlassian.crowd.exception.ApplicationNotFoundException;
import com.atlassian.crowd.exception.InvalidAuthenticationException;
import com.atlassian.crowd.exception.InvalidEmailAddressException;
import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.crowd.exception.OperationNotPermittedException;
import com.atlassian.crowd.exception.UserNotFoundException;
import com.atlassian.crowd.manager.mail.MailSendException;

/**
 * Used to change e-mail of a user in a way that they need to prove that they have access to the e-mail they're trying
 * to change to. This is achieved by sending an e-mail message containing a secret. Only and only if
 * a user is able to bring the secret back to Crowd their e-mail is being changed.
 *
 * @since 4.4.0
 */
public interface EmailChangeManager {
    /**
     * Starts e-mail change flow for a user in crowd app, by sending two e-mails. First one to the current user's
     * e-mail and the second one to the <code>newEmail</code>. The first one contains info for an ongoing e-mail change.
     * The second one contains a link with a token which is required to change an e-mail. In order to do change an
     * e-mail, {@link EmailChangeManager#changeEmail} needs to be called with the aforementioned token passed as
     * an argument.
     *
     * @param username    username of the user for whom we're changing an e-mail
     * @param password    password of the user for whom we're changing an e-mail
     * @param directoryId directory ID of the user for whom we're changing an e-mail
     * @param newEmail    e-mail to which we'll replace an old email of the user
     * @throws InvalidAuthenticationException if authentication fails for given {@code username} and {@code password}
     * @throws InvalidEmailAddressException if passed {@code newEmail} is not a valid e-mail address
     * @throws SameEmailAddressException if passed {@code newEmail} is the same as the one currently used by a user
     * @throws MailSendException if Crowd failed to send e-mail with a token
     * @throws OperationNotPermittedException if feature is not available
     * @throws ApplicationNotFoundException if 'crowd' app has not been found
     */
    void sendEmailAuthorization(String username, String password, long directoryId, String newEmail)
            throws InvalidAuthenticationException, InvalidEmailAddressException,
            MailSendException, OperationNotPermittedException, ApplicationNotFoundException, SameEmailAddressException;

    /**
     * Changes e-mail for a user in crowd app.
     * <p>
     * This method SHALL only be called after calling {@link EmailChangeManager#sendEmailAuthorization}.
     *
     * @param token    token sent in email by {@link EmailChangeManager#sendEmailAuthorization}
     * @param username username of the user for whom we're changing an e-mail
     * @throws InvalidChangeEmailTokenException if provide {@code token} is not valid
     * @throws InvalidAuthenticationException   if user identified by {@code username} is not the one who generated a token
     * @throws UserNotFoundException            if user does not exist
     * @throws OperationFailedException         if Crowd failed to update e-mail
     * @throws ApplicationNotFoundException     if 'crowd' app has not been found
     * @throws OperationNotPermittedException   if feature is not available
     */
    void changeEmail(String token, String username) throws InvalidChangeEmailTokenException, InvalidAuthenticationException,
            UserNotFoundException, OperationFailedException, ApplicationNotFoundException, OperationNotPermittedException;

    /**
     * Returns true if there is a pending e-mail verification for a user in crowd app, false otherwise.
     *
     * @param username    username of the user
     * @param directoryId directory ID of the user
     * @return true if there is a pending e-mail verification for a user, false otherwise
     */
    boolean hasUserEmailChangePending(String username, long directoryId);

    /**
     * Returns new e-mail address for a user in crowd app for whom there's a pending e-mail verification.
     *
     * @param username    username of the user
     * @param directoryId directory ID of the user
     * @return new e-mail address
     * @throws IllegalStateException if user does not have a pending e-mail validation
     */
    String getPendingNewEmailByUser(String username, long directoryId);

    /**
     * Returns new e-mail address for a e-mail change token.
     *
     * @param token e-mail change token
     * @return new e-mail address
     * @throws IllegalStateException if user does not have a pending e-mail validation
     */
    String getPendingNewEmailByToken(String token);

    /**
     * Resends an e-mail with a new token for a user in crowd app that has a pending e-mail verification. Old token is
     * invalidated.
     *
     * @param username    of the user
     * @param directoryId directory ID of the user
     * @throws IllegalStateException        if user does not have a pending e-mail validation
     * @throws InvalidEmailAddressException if pending e-mail address is not a valid e-mail address
     * @throws MailSendException            if Crowd failed to re-send e-mail with a token
     * @throws UserNotFoundException        if user does not exist
     * @throws ApplicationNotFoundException if 'crowd' app has not been found
     */
    void resendEmail(String username, long directoryId) throws InvalidEmailAddressException,
            MailSendException, UserNotFoundException, ApplicationNotFoundException;

    /**
     * Aborts a pending e-mail validation for a given user in crowd app.
     *
     * @param username    username of the user
     * @param directoryId directory ID of the user
     * @throws IllegalStateException if user does not have a pending e-mail validation
     */
    void abort(String username, long directoryId);

    /**
     * Returns true if a given token is a valid e-mail change verification token. False otherwise.
     *
     * @param token e-mail change token
     * @return true if a given token is a valid e-mail change verification token, false otherwise
     */
    boolean isTokenValid(String token);

    /**
     * Returns true if e-mail change by end-user feature is available for a given directory. False otherwise.
     * <p>
     * Feature might be disabled for a given directory because:
     * - Crowd's e-mail server is NOT configured
     * - Directory does not have permissions to change user e-mail
     *
     * @param dirId ID of the directory
     * @return true if e-mail change by end-user feature is available for a given directory, false otherwise.
     */
    boolean isAvailableForDirectory(long dirId);
}
