package com.atlassian.crowd.manager.permission;

import com.atlassian.annotations.ExperimentalApi;
import com.atlassian.crowd.exception.DirectoryNotFoundException;
import com.atlassian.crowd.exception.FeatureInaccessibleException;
import com.atlassian.crowd.exception.GroupNotFoundException;
import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.crowd.exception.UserNotFoundException;
import com.atlassian.crowd.model.group.Group;
import com.atlassian.crowd.model.user.ImmutableUser;
import com.atlassian.crowd.model.user.User;

import javax.annotation.Nonnull;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * Allows clients to check group permissions of users
 * @since 3.3.0
 */
@ExperimentalApi
public interface GroupAdministrationPermissionService {

    /**
     * Looks for groups that currently logged in user administers
     *
     * @param query the criteria that the returned groups must meet or null for all groups
     * @return a set of mappings containing all groups currently logged in user administers
     * @throws IllegalStateException if user associated with the current session, no more exists in the directory
     */
    List<UserGroupAdministrationMapping> getAdministeredGroupsForCurrentUser(@Nonnull AdministeredGroupsQuery query) throws OperationFailedException, FeatureInaccessibleException;

    /**
     * Looks for groups a user administers
     * @param user the user
     * @param query the criteria that the returned groups must meet or null for all groups
     * @return a set of mappings containing all groups user can administer
     * @throws IllegalStateException if the supplied user doesn't exists
     */
    List<UserGroupAdministrationMapping> getAdministeredGroups(User user, @Nonnull AdministeredGroupsQuery query) throws OperationFailedException, FeatureInaccessibleException;

    /**
     * @param user the user
     * @return true if the specified user administers at least one group, either through a directly assigned grant or
     * through memberships
     */
    boolean isUserGroupLevelAdmin(User user) throws OperationFailedException;

    /**
     * Check if a current logged in user is an administrator of a group
     * @param group the group
     * @return {@code true} if the user is an administrator of the group, {@code false} otherwise
     */
    boolean isCurrentUserAdminOfGroup(Group group) throws GroupNotFoundException, UserNotFoundException, DirectoryNotFoundException, OperationFailedException, FeatureInaccessibleException;

    /**
     * Check if a user is an administrator of a group
     * @param user the user
     * @param group the group
     * @return {@code true} if the user is an administrator of the group, {@code false} otherwise
     */
    boolean isUserAdminOfGroup(User user, Group group) throws GroupNotFoundException, UserNotFoundException, DirectoryNotFoundException, OperationFailedException, FeatureInaccessibleException;

    /**
     * @param considerOnlyTargetGroupsWithApplications whether only groups with mapped application should be taken into account
     * @return set of users with direct grants to manage any group
     */
    Set<User> getDirectGroupLevelAdminUsers(boolean considerOnlyTargetGroupsWithApplications);

    /**
     * @param considerOnlyTargetGroupsWithApplications whether only groups with mapped application should be taken into account
     * @return set of groups with direct grants to manage any group
     */
    Set<Group> getDirectGroupLevelAdminGroups(boolean considerOnlyTargetGroupsWithApplications);
}
