package com.atlassian.crowd.model.token;

import javax.annotation.Nullable;
import java.io.Serializable;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Represent a token that can expire for a user
 *
 * @since v2.8
 */
public final class InternalExpirableUserToken implements Serializable, ExpirableUserToken {
    private long id;
    private String token;
    @Nullable
    private String username;
    @Nullable
    private String emailAddress;
    private long expiryDate;
    private long directoryId;
    private ExpirableUserTokenType type;

    InternalExpirableUserToken() {
        // For hibernate
    }

    public static InternalExpirableUserToken createResetPasswordToken(final String token, final String username,
                                                                      final String email, final long expiryDate,
                                                                      final long directoryId) {
        return new InternalExpirableUserToken(token, username, email, expiryDate, directoryId,
                ExpirableUserTokenType.UNSPECIFIED);
    }

    public static InternalExpirableUserToken createInviteUserToken(final String token, final String email,
                                                                   final long expiryDate, final long directoryId) {
        return new InternalExpirableUserToken(token, null, email, expiryDate, directoryId,
                ExpirableUserTokenType.UNSPECIFIED);
    }

    public static InternalExpirableUserToken createChangeEmailToken(String token, String username,
                                                                    String newEmailAddress, long expiryDate,
                                                                    long directoryId) {
        return new InternalExpirableUserToken(token, username, newEmailAddress, expiryDate, directoryId,
                ExpirableUserTokenType.VALIDATE_EMAIL);
    }

    private InternalExpirableUserToken(final String token, @Nullable final String username,
                                       @Nullable final String emailAddress, final long expiryDate,
                                       final long directoryId, ExpirableUserTokenType type) {
        this.token = checkNotNull(token, "token must not be null");
        this.username = username;
        this.emailAddress = emailAddress;
        this.expiryDate = expiryDate;
        this.directoryId = directoryId;
        this.type = type;
    }

    public long getId() {
        return id;
    }

    private void setId(final long id) {
        this.id = id;
    }

    @Override
    public String getToken() {
        return token;
    }

    private void setToken(final String token) {
        this.token = checkNotNull(token, "token must not be null");
    }

    @Nullable
    @Override
    public String getUsername() {
        return username;
    }

    private void setUsername(@Nullable final String username) {
        this.username = username;
    }

    @Nullable
    @Override
    public String getEmailAddress() {
        return emailAddress;
    }

    private void setEmailAddress(final String emailAddress) {
        this.emailAddress = emailAddress;
    }

    @Override
    public long getExpiryDate() {
        return expiryDate;
    }

    private void setExpiryDate(final long expiryDate) {
        this.expiryDate = expiryDate;
    }

    @Override
    public long getDirectoryId() {
        return directoryId;
    }

    private void setDirectoryId(final long directoryId) {
        this.directoryId = directoryId;
    }

    @Override
    public ExpirableUserTokenType getType() {
        return type;
    }

    public void setType(ExpirableUserTokenType type) {
        this.type = type;
    }

    @Override
    public boolean equals(final Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        final InternalExpirableUserToken that = (InternalExpirableUserToken) o;

        if (token != null ? !token.equals(that.token) : that.token != null) {
            return false;
        }

        return true;
    }

    @Override
    public int hashCode() {
        return (token != null) ? token.hashCode() : 0;
    }
}
