package com.atlassian.crowd.dao.membership.cache;

import com.atlassian.crowd.dao.membership.InternalMembershipDao;
import com.atlassian.crowd.embedded.api.SearchRestriction;
import com.atlassian.crowd.embedded.impl.IdentifierMap;
import com.atlassian.crowd.embedded.impl.IdentifierUtils;
import com.atlassian.crowd.embedded.spi.MembershipDao;
import com.atlassian.crowd.exception.GroupNotFoundException;
import com.atlassian.crowd.exception.MembershipAlreadyExistsException;
import com.atlassian.crowd.exception.MembershipNotFoundException;
import com.atlassian.crowd.exception.UserNotFoundException;
import com.atlassian.crowd.model.NameComparator;
import com.atlassian.crowd.model.group.GroupType;
import com.atlassian.crowd.search.Entity;
import com.atlassian.crowd.search.query.entity.EntityQuery;
import com.atlassian.crowd.search.query.entity.restriction.NullRestriction;
import com.atlassian.crowd.search.query.membership.MembershipQuery;
import com.atlassian.crowd.search.util.SearchResultsUtil;
import com.atlassian.crowd.util.BatchResult;
import com.atlassian.crowd.util.BoundedCount;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableListMultimap;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Iterables;
import com.google.common.collect.ListMultimap;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * Caching wrapper over {@link InternalMembershipDao}.
 */
public class CachingMembershipDao implements MembershipDao {
    private final InternalMembershipDao delegate;
    protected final MembershipCache membershipCache;

    public CachingMembershipDao(InternalMembershipDao delegate,
                                MembershipCache membershipCache) {
        this.delegate = delegate;
        this.membershipCache = membershipCache;
    }

    @Override
    public boolean isUserDirectMember(long directoryId, String userName, String groupName) {
        List<String> groupUsers = membershipCache.getNames(directoryId, QueryType.GROUP_USERS, groupName);
        if (groupUsers != null) {
            return containsLowerCase(groupUsers, userName);
        }
        List<String> userGroups = membershipCache.getNames(directoryId, QueryType.USER_GROUPS, userName);
        if (userGroups != null) {
            return containsLowerCase(userGroups, groupName);
        }
        return delegate.isUserDirectMember(directoryId, userName, groupName);
    }

    @Override
    public boolean isGroupDirectMember(long directoryId, String childGroup, String parentGroup) {
        List<String> children = membershipCache.getNames(directoryId, QueryType.GROUP_SUBGROUPS, parentGroup);
        if (children != null) {
            return containsLowerCase(children, childGroup);
        }
        List<String> parents = membershipCache.getNames(directoryId, QueryType.GROUP_PARENTS, childGroup);
        if (parents != null) {
            return containsLowerCase(parents, parentGroup);
        }
        return delegate.isGroupDirectMember(directoryId, childGroup, parentGroup);
    }

    @Override
    public void addUserToGroup(long directoryId, String userName, String groupName) throws UserNotFoundException, GroupNotFoundException, MembershipAlreadyExistsException {
        membershipCache.invalidateCache(directoryId, QueryType.USER_GROUPS, userName);
        membershipCache.invalidateCache(directoryId, QueryType.GROUP_USERS, groupName);
        delegate.addUserToGroup(directoryId, userName, groupName);
    }

    @Override
    public BatchResult<String> addAllUsersToGroup(long directoryId, Collection<String> userNames, String groupName) throws GroupNotFoundException {
        membershipCache.invalidateCache(directoryId, QueryType.GROUP_USERS, groupName);
        userNames.forEach(userName -> membershipCache.invalidateCache(directoryId, QueryType.USER_GROUPS, userName));
        return delegate.addAllUsersToGroup(directoryId, userNames, groupName);
    }

    @Override
    public void addGroupToGroup(long directoryId, String childGroup, String parentGroup) throws GroupNotFoundException, MembershipAlreadyExistsException {
        membershipCache.invalidateCache(directoryId, QueryType.GROUP_PARENTS, childGroup);
        membershipCache.invalidateCache(directoryId, QueryType.GROUP_SUBGROUPS, parentGroup);
        delegate.addGroupToGroup(directoryId, childGroup, parentGroup);
    }

    @Override
    public void removeUserFromGroup(long directoryId, String userName, String groupName) throws UserNotFoundException, GroupNotFoundException, MembershipNotFoundException {
        membershipCache.invalidateCache(directoryId, QueryType.USER_GROUPS, userName);
        membershipCache.invalidateCache(directoryId, QueryType.GROUP_USERS, groupName);
        delegate.removeUserFromGroup(directoryId, userName, groupName);
    }

    @Override
    public void removeGroupFromGroup(long directoryId, String childGroup, String parentGroup) throws GroupNotFoundException, MembershipNotFoundException {
        membershipCache.invalidateCache(directoryId, QueryType.GROUP_PARENTS, childGroup);
        membershipCache.invalidateCache(directoryId, QueryType.GROUP_SUBGROUPS, parentGroup);
        delegate.removeGroupFromGroup(directoryId, childGroup, parentGroup);
    }

    @Override
    public <T> List<T> search(long directoryId, MembershipQuery<T> query) {
        if (shouldCache(query)) {
            ListMultimap<String, T> results = searchGroupedByNameCached(directoryId, query.withAllResults());
            if (results.isEmpty()) {
                return ImmutableList.of();
            } else if (results.keySet().size() == 1) {
                List<T> allResults = results.get(Iterables.getOnlyElement(results.keySet()));
                return SearchResultsUtil.constrainResults(allResults, query.getStartIndex(), query.getMaxResults());
            }
            return indexByName(results.values())
                    .entrySet()
                    .stream()
                    .sorted(Map.Entry.comparingByKey())
                    .map(Map.Entry::getValue)
                    .skip(query.getStartIndex())
                    .limit(EntityQuery.allResultsToLongMax(query.getMaxResults()))
                    .collect(Collectors.toList());
        } else {
            return delegate.search(directoryId, query);
        }
    }

    private <T> Map<String, T> indexByName(Collection<T> values) {
        if (values.isEmpty()) {
            return ImmutableMap.of();
        }
        final Function<T, String> normalizer = NameComparator.normaliserOf((Class<T>) values.iterator().next().getClass());
        Map<String, T> results = new HashMap<>();
        values.forEach(e -> results.putIfAbsent(normalizer.apply(e), e));
        return results;
    }

    @Override
    public <T> ListMultimap<String, T> searchGroupedByName(long directoryId, MembershipQuery<T> query) {
        if (shouldCache(query)) {
            return searchGroupedByNameCached(directoryId, query);
        }
        return delegate.searchGroupedByName(directoryId, query);
    }

    private <T> ListMultimap<String, T> searchGroupedByNameCached(long directoryId, MembershipQuery<T> query) {
        QueryType queryType = getQueryType(query);
        ListMultimap<String, T> resultsByName = ArrayListMultimap.create();
        final Set<String> missing = new HashSet<>();
        for (String name : query.getEntityNamesToMatch()) {
            List<T> results = membershipCache.get(directoryId, queryType, name, query.getReturnType());
            if (results == null) {
                missing.add(name);
            } else {
                resultsByName.putAll(name, results);
            }
        }
        if (!missing.isEmpty()) {
            resultsByName.putAll(executeAndCache(directoryId, query.withEntityNames(missing).withAllResults()));
        }
        return resultsByName;
    }

    private <T> ListMultimap<String, T> executeAndCache(long directoryId, MembershipQuery<T> query) {
        QueryType queryType = getQueryType(query);
        ListMultimap<String, T> results = doSearchGroupedByName(directoryId, query);

        IdentifierMap<Collection<T>> identityMap = new IdentifierMap<>(results.asMap());
        for (String key : query.getEntityNamesToMatch()) {
            ImmutableList<T> values = ImmutableList.copyOf(identityMap.getOrDefault(key, ImmutableList.of()));
            membershipCache.put(directoryId, queryType, key, values);
        }
        return results;
    }

    private <T> ListMultimap<String, T> doSearchGroupedByName(long directoryId, MembershipQuery<T> query) {
        if (query.getEntityNamesToMatch().size() == 1) {
            return ImmutableListMultimap.<String, T>builder()
                    .putAll(Iterables.getOnlyElement(query.getEntityNamesToMatch()), delegate.search(directoryId, query))
                    .build();
        } else {
            return delegate.searchGroupedByName(directoryId, query);
        }
    }

    @Override
    public BoundedCount countDirectMembersOfGroup(long directoryId, String groupName, int potentialMaxCount) {
        List<String> cached = membershipCache.getNames(directoryId, QueryType.GROUP_USERS, groupName);
        return cached != null ? BoundedCount.exactly(cached.size()) :
                delegate.countDirectMembersOfGroup(directoryId, groupName, potentialMaxCount);
    }

    @Override
    public BatchResult<String> addUserToGroups(long directoryId, String username, Set<String> groupNames) throws UserNotFoundException {
        membershipCache.invalidateCache(directoryId, QueryType.USER_GROUPS, username);
        groupNames.forEach(name -> membershipCache.invalidateCache(directoryId, QueryType.GROUP_USERS, name));
        return delegate.addUserToGroups(directoryId, username, groupNames);
    }

    protected <T> boolean shouldCache(MembershipQuery<T> query) {
        SearchRestriction restriction = query.getSearchRestriction();
        if (restriction != null && !(restriction instanceof NullRestriction)) {
            return false;
        }
        if (!membershipCache.supports(query.getReturnType()) && !query.isWithAllResults()) {
            return false;
        }
        if (query.getEntityToMatch().getGroupType() != null && query.getEntityToMatch().getGroupType() != GroupType.GROUP) {
            return false;
        }
        if (query.getEntityToReturn().getGroupType() != null && query.getEntityToReturn().getGroupType() != GroupType.GROUP) {
            return false;
        }
        return membershipCache.getCacheableTypes().contains(getQueryType(query));
    }

    private <T> QueryType getQueryType(MembershipQuery<T> query) {
        if (query.isFindChildren()) {
            return query.getEntityToReturn().getEntityType() == Entity.USER ? QueryType.GROUP_USERS : QueryType.GROUP_SUBGROUPS;
        } else {
            return query.getEntityToMatch().getEntityType() == Entity.USER ? QueryType.USER_GROUPS : QueryType.GROUP_PARENTS;
        }
    }

    private boolean containsLowerCase(Collection<String> values, String toMatch) {
        String toMatchLowercase = IdentifierUtils.toLowerCase(toMatch);
        return values.stream().anyMatch(v -> IdentifierUtils.toLowerCase(v).equals(toMatchLowercase));
    }

    public void clearCache() {
        membershipCache.clear();
    }

    // TODO handle XMLRestoreFinishedEvent
}
