package com.atlassian.crowd.directory;

import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.manager.directory.SynchronisationMode;
import com.atlassian.crowd.model.directory.DirectorySynchronisationStatus;
import com.atlassian.crowd.model.directory.SynchronisationStatusKey;
import com.google.common.base.MoreObjects;
import com.google.common.base.Objects;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.Serializable;
import java.util.List;

/**
 * Implementation of {@code com.atlassian.crowd.model.directory.DirectorySynchronisationStatus}
 *
 * @since 2.12.0
 */
public class DirectorySynchronisationStatusImpl implements DirectorySynchronisationStatus {
    private Integer id;
    private Directory directory;
    private Long startTimestamp;
    private Long endTimestamp;
    private SynchronisationStatusKey status;
    private String statusParameters;
    private String incrementalSyncError;
    private String fullSyncError;
    private String nodeId;
    private String nodeName;

    public DirectorySynchronisationStatusImpl() {

    }

    protected DirectorySynchronisationStatusImpl(final Builder builder) {
        this.id = builder.id;
        this.directory = builder.directory;
        this.startTimestamp = builder.startTimestamp;
        this.endTimestamp = builder.endTimestamp;
        this.status = builder.status;
        this.statusParameters = builder.statusParameters;
        this.incrementalSyncError = builder.incrementalSyncError;
        this.fullSyncError = builder.fullSyncError;
        this.nodeId = builder.nodeId;
        this.nodeName = builder.nodeName;
    }

    @Override
    public Integer getId() {
        return id;
    }

    public void setId(Integer id) {
        this.id = id;
    }

    @Override
    public Directory getDirectory() {
        return directory;
    }

    public void setDirectory(Directory directory) {
        this.directory = directory;
    }

    @Override
    public long getStartTimestamp() {
        return startTimestamp;
    }

    public void setStartTimestamp(long start) {
        this.startTimestamp = start;
    }

    @Override
    public Long getEndTimestamp() {
        return endTimestamp;
    }

    public void setEndTimestamp(Long end) {
        this.endTimestamp = end;
    }

    @Override
    public SynchronisationStatusKey getStatus() {
        return status;
    }

    public void setStatus(SynchronisationStatusKey status) {
        this.status = status;
    }

    @Override
    public String getStatusParameters() {
        return statusParameters;
    }

    public void setStatusParameters(String statusParameters) {
        this.statusParameters = statusParameters;
    }

    @Override
    public String getIncrementalSyncError() {
        return incrementalSyncError;
    }

    public void setIncrementalSyncError(String incrementalSyncError) {
        this.incrementalSyncError = incrementalSyncError;
    }

    @Override
    public String getFullSyncError() {
        return fullSyncError;
    }

    public void setFullSyncError(String fullSyncError) {
        this.fullSyncError = fullSyncError;
    }

    @Override
    public String getNodeId() {
        return nodeId;
    }

    public void setNodeId(String nodeId) {
        this.nodeId = nodeId;
    }

    @Override
    public String getNodeName() {
        return nodeName;
    }

    public void setNodeName(String nodeName) {
        this.nodeName = nodeName;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        DirectorySynchronisationStatusImpl that = (DirectorySynchronisationStatusImpl) o;
        return Objects.equal(id, that.id) &&
                Objects.equal(directory, that.directory) &&
                Objects.equal(startTimestamp, that.startTimestamp) &&
                Objects.equal(endTimestamp, that.endTimestamp) &&
                status == that.status &&
                Objects.equal(statusParameters, that.statusParameters) &&
                Objects.equal(incrementalSyncError, that.incrementalSyncError) &&
                Objects.equal(fullSyncError, that.fullSyncError) &&
                Objects.equal(nodeId, that.nodeId) &&
                Objects.equal(nodeName, that.nodeName);
    }

    @Override
    public int hashCode() {
        return Objects.hashCode(id, directory, startTimestamp, endTimestamp, status, statusParameters, incrementalSyncError, fullSyncError, nodeId, nodeName);
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("id", id)
                .add("directory", directory.getId())
                .add("startTimestamp", startTimestamp)
                .add("endTimestamp", endTimestamp)
                .add("status", status)
                .add("statusParameters", statusParameters)
                .add("incrementalSyncError", incrementalSyncError)
                .add("fullSyncError", fullSyncError)
                .add("nodeId", nodeId)
                .add("nodeName", nodeName)
                .toString();
    }

    public static Builder builder() {
        return new Builder();
    }

    public static Builder builder(DirectorySynchronisationStatus status) {
        return new Builder(status);
    }

    public static class Builder {
        private static final Logger logger = LoggerFactory.getLogger(Builder.class);

        private Integer id;
        private Directory directory;
        private Long startTimestamp;
        private Long endTimestamp;
        private SynchronisationStatusKey status;
        private String statusParameters;
        private String incrementalSyncError;
        private String fullSyncError;
        private String nodeId;
        private String nodeName;

        public Builder() {
        }

        public Builder(DirectorySynchronisationStatus status) {
            this.id = status.getId();
            this.directory = status.getDirectory();
            this.startTimestamp = status.getStartTimestamp();
            this.endTimestamp = status.getEndTimestamp();
            this.status = status.getStatus();
            this.statusParameters = status.getStatusParameters();
            this.incrementalSyncError = status.getIncrementalSyncError();
            this.fullSyncError = status.getFullSyncError();
            this.nodeId = status.getNodeId();
            this.nodeName = status.getNodeName();
        }

        public Builder setId(Integer id) {
            this.id = id;
            return this;
        }

        public Builder setDirectory(Directory directory) {
            this.directory = directory;
            return this;
        }

        public Builder setStartTimestamp(Long startTimestamp) {
            this.startTimestamp = startTimestamp;
            return this;
        }

        public Builder setEndTimestamp(Long endTimestamp) {
            this.endTimestamp = endTimestamp;
            return this;
        }

        public Builder setStatus(SynchronisationStatusKey status, List<Serializable> statusParameters) {
            this.status = status;
            this.statusParameters = status.marshallParams(statusParameters);
            return this;
        }

        public Builder setStatus(SynchronisationStatusKey status, String statusParameters) {
            this.status = status;
            this.statusParameters = statusParameters;
            return this;
        }

        public Builder setSyncError(SynchronisationMode syncMode, String errorMessage) {
            switch (syncMode) {
                case FULL:
                    fullSyncError = errorMessage;
                    break;
                case INCREMENTAL:
                    incrementalSyncError = errorMessage;
                    break;
                default:
                    logger.warn("Got synchronization error for mode {} that is not supported. Skipping: {}", syncMode, errorMessage);
            }
            return this;
        }

        public Builder setNodeId(String nodeId) {
            this.nodeId = nodeId;
            return this;
        }

        public Builder setNodeName(String nodeName) {
            this.nodeName = nodeName;
            return this;
        }

        public DirectorySynchronisationStatusImpl build() {
            return new DirectorySynchronisationStatusImpl(this);
        }
    }
}
