package com.atlassian.crowd.dao.tombstone;

import java.util.Collection;
import java.util.List;

import com.atlassian.crowd.model.tombstone.AbstractTombstone;
import com.atlassian.crowd.model.tombstone.ApplicationTombstone;
import com.atlassian.crowd.model.tombstone.ApplicationUpdatedTombstone;

/**
 * Allows storing tombstones - persistent objects that denote a deletion of an entity or state.
 * Note that tombstones are pruned after a time by a background job (see TombstoneReaper)
 */
public interface TombstoneDao {
    /**
     * Stores a tombstone for users with the given names
     *
     * @param directoryId the directory id
     * @param names       the names of the users to store the tombstone for
     */
    void storeUserTombstones(long directoryId, Collection<String> names);

    /**
     * Stores a tombstone for groups with the given names
     *
     * @param directoryId the directory id
     * @param names       the names of the groups to store the tombstone for
     */
    void storeGroupTombstones(long directoryId, Collection<String> names);

    /**
     * Stores a tombstone for user-group memberships
     *
     * @param directoryId     the directory id
     * @param username        the name of the user
     * @param parentGroupName the name of the group the user belonged to
     */
    void storeUserMembershipTombstone(long directoryId, String username, String parentGroupName);

    /**
     * Stores a tombstone for group-group memberships
     *
     * @param directoryId     the directory id
     * @param childGroupName  the name of the user
     * @param parentGroupName the name of the group the user belonged to
     */
    void storeGroupMembershipTombstone(long directoryId, String childGroupName, String parentGroupName);

    /**
     * Stores a tombstone denoting events for the given directory can't be reconstructed
     *
     * @param reason      describes why this tombstone was written
     * @param directoryId the directory id this affects
     * @see com.atlassian.crowd.event.TimestampBasedEventStore
     */
    void storeEventsTombstoneForDirectory(String reason, long directoryId);

    /**
     * Stores a tombstone denoting events for the given application can't be reconstructed
     *
     * @param applicationId the application id this affects
     * @see com.atlassian.crowd.event.TimestampBasedEventStore
     */
    void storeEventsTombstoneForApplication(long applicationId);

    /**
     * Stores a tombstone denoting any events can't be reconstructed
     *
     * @param reason describes why this tombstone was written
     * @see com.atlassian.crowd.event.TimestampBasedEventStore
     */
    void storeEventsTombstone(String reason);

    /**
     * Stores a tombstone for an alias
     *
     * @param applicationId the application id this affects
     * @param username      the original user name for the alias
     */
    void storeAliasTombstone(long applicationId, String username);

    /**
     * Retrieves tombstone of the specified type, after the timestamp specified, optionally constrained to a directory set
     *
     * @param after        only tombstones persisted after this timestamp will be returned
     * @param directoryIds if empty all tombstones will be returned, otherwise only tombstones for the specified directories.
     *                     Should be when retrieving tombstones that aren't directory specific.
     * @param type         only retrieve tombstone of the specified type. Use {@link AbstractTombstone} to search for all types.
     * @return a list of tombstones matching the criteria
     */
    <T extends AbstractTombstone> List<T> getTombstonesAfter(long after, Collection<Long> directoryIds, Class<T> type);

    /**
     * Retrieves tombstone for the specified application, after the timestamp specified
     *
     * @param after        only tombstones persisted after this timestamp will be returned
     * @param applicationId if null all tombstones will be returned, otherwise only tombstones for the specified application.
     *                     Should be when retrieving tombstones that aren't directory specific.
     * @return a list of tombstones matching the criteria
     */
    <T extends ApplicationTombstone> List<T> getTombstonesAfter(long after, Long applicationId, Class<T> type);

    /**
     * Removes all tombstones with the timestamp less or equal than the specified one.
     *
     * @return the number of tombstones removed
     */
    int removeAllUpTo(long timestamp);
}