package com.atlassian.crowd.dao.membership.cache;

import com.atlassian.crowd.embedded.impl.IdentifierUtils;
import com.google.common.collect.ImmutableSet;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

/**
 * Collection of membership cache invalidations. 
 * Invalidations within a single transaction are aggregated and performed after transaction is committed.
 */
public class CacheInvalidations {
    private final Set<QueryTypeCacheKey> queryTypesInvalidations = new HashSet<>();
    private final Map<QueryTypeCacheKey, Set<String>> keyInvalidations = new HashMap<>();
    private final Set<QueryType> cacheableTypes;
    private final int queryTypeInvalidationThreshold;

    /**
     * @param cacheableTypes                 Cacheable query types
     * @param queryTypeInvalidationThreshold when number of invalidated keys goes above this threshold then all entries
     *                                       for the given query type will be invalidated; this allows to reduce number
     *                                       of invalidation messages sent to the cluster
     */
    public CacheInvalidations(Set<QueryType> cacheableTypes, int queryTypeInvalidationThreshold) {
        this.cacheableTypes = ImmutableSet.copyOf(cacheableTypes);
        this.queryTypeInvalidationThreshold = queryTypeInvalidationThreshold;
    }

    public boolean isInvalidated(QueryTypeCacheKey cacheKey, String key) {
        if (queryTypesInvalidations.contains(cacheKey)) {
            return true;
        }
        Set<String> invalidatedKeys = keyInvalidations.get(cacheKey);
        return invalidatedKeys != null && invalidatedKeys.contains(IdentifierUtils.toLowerCase(key));
    }

    public void addInvalidation(long directoryId) {
        for (QueryType type : cacheableTypes) {
            addInvalidation(directoryId, type);
        }
    }

    public void addInvalidation(long directoryId, QueryType queryType) {
        QueryTypeCacheKey cacheKey = new QueryTypeCacheKey(directoryId, queryType);
        queryTypesInvalidations.add(cacheKey);
        keyInvalidations.remove(cacheKey);
    }

    public void addInvalidation(long directoryId, QueryType queryType, String key) {
        QueryTypeCacheKey cacheKey = new QueryTypeCacheKey(directoryId, queryType);
        if (!queryTypesInvalidations.contains(cacheKey)) {
            Set<String> keys = keyInvalidations.computeIfAbsent(cacheKey, k -> new HashSet<>());
            keys.add(IdentifierUtils.toLowerCase(key));
            if (keys.size() > queryTypeInvalidationThreshold) {
                queryTypesInvalidations.add(cacheKey);
                keyInvalidations.remove(cacheKey);
            }
        }
    }

    public Set<QueryTypeCacheKey> getQueryTypesInvalidations() {
        return queryTypesInvalidations;
    }
    public Map<QueryTypeCacheKey, Set<String>> getKeyInvalidations() {
        return keyInvalidations;
    }
}
