package com.atlassian.crowd.dao.application;

import com.atlassian.crowd.exception.ApplicationNotFoundException;
import com.atlassian.crowd.exception.DirectoryMappingNotFoundException;
import com.atlassian.crowd.model.application.Application;
import com.atlassian.crowd.model.application.ApplicationDirectoryMapping;
import com.atlassian.crowd.model.application.InternalApplicationDefaultGroupMembershipConfiguration;

import java.util.List;

/**
 * Hibernate DAO for managing application default group memberships
 *
 * @since v3.1.0
 */
public interface ApplicationDefaultGroupMembershipConfigurationDao {

    /**
     * Adds a new default group membership for the given application and group.
     * @param application the application for which the default group membership will be created
     * @param directoryMapping directory mapping between the application and the directory from which the group originates
     * @param groupName the name of the group in the default group membership configuration
     * @throws DirectoryMappingNotFoundException when the directory mapping could not be found
     */
    void add(Application application, ApplicationDirectoryMapping directoryMapping, String groupName) throws DirectoryMappingNotFoundException, ApplicationNotFoundException;

    /**
     * Removes a default group membership configuration for the given application and group.
     * @param application the application for which the default group membership will be removed
     * @param directoryMapping directory mapping between the application and the directory from which the group originates
     * @param groupName the name of the group in the default group membership configuration
     * @throws DirectoryMappingNotFoundException when the directory mapping could not be found
     */
    void remove(Application application, ApplicationDirectoryMapping directoryMapping, String groupName) throws DirectoryMappingNotFoundException, ApplicationNotFoundException;

    /**
     * Removes all default group membership configurations for the given application and directory combination.
     * @param application the application for which the default group memberships will be removed
     * @param directoryMapping directory mapping between the application and the directory from which the group originates
     * @throws DirectoryMappingNotFoundException when the directory mapping could not be found
     */
    void removeAll(Application application, ApplicationDirectoryMapping directoryMapping) throws DirectoryMappingNotFoundException, ApplicationNotFoundException;

    /**
     * Lists all default group membership configurations for the given application/directory combination
     * @param application the application for which the default group memberships should be listed
     * @param directoryMapping directory mapping between the application and the directory from which the group originates
     * @throws DirectoryMappingNotFoundException when the directory mapping could not be found
     * @return a list of default group memberships configurations for the given application/directory
     * combination
     */
    List<InternalApplicationDefaultGroupMembershipConfiguration> listAll(Application application, ApplicationDirectoryMapping directoryMapping) throws DirectoryMappingNotFoundException, ApplicationNotFoundException;
}
