package com.atlassian.crowd.crypto;

import com.atlassian.crowd.dao.property.PropertyDAO;
import com.atlassian.crowd.embedded.api.DataReEncryptor;
import com.atlassian.crowd.exception.ObjectNotFoundException;
import com.atlassian.crowd.model.property.Property;

import java.util.List;
import java.util.function.UnaryOperator;
import java.util.stream.Collectors;

public class EncryptingPropertyDAO implements PropertyDAO, DataReEncryptor {
    private PropertyDAO delegate;
    private PropertyEncryptor encryptor;

    public void setDelegate(PropertyDAO delegate) {
        this.delegate = delegate;
    }

    public void setEncryptor(PropertyEncryptor encryptor) {
        this.encryptor = encryptor;
    }

    @Override
    public Property find(String key, String name) throws ObjectNotFoundException {
        return encryptor.decrypt(delegate.find(key, name));
    }

    @Override
    public List<Property> findAll(String key) {
        return delegate.findAll(key).stream().map(encryptor::decrypt).collect(Collectors.toList());
    }

    @Override
    public Property add(Property property) {
        return addOrUpdate(delegate::add, property);
    }

    @Override
    public Property update(Property property) {
        return addOrUpdate(delegate::update, property);
    }

    private Property addOrUpdate(UnaryOperator<Property> operation, Property property) {
        final Property encrypted = encryptor.encrypt(property);
        final Property addedOrUpdated = operation.apply(encrypted);
        return encryptor.decrypt(addedOrUpdated);
    }

    @Override
    public void remove(String key, String name) {
        delegate.remove(key, name);
    }

    @Override
    public List<Property> findAll() {
        return delegate.findAll().stream().map(encryptor::decrypt).collect(Collectors.toList());
    }

    @Override
    public void reEncrypt() {
        findAll().forEach(this::update);
    }
}
