package com.atlassian.crowd.directory;

import java.util.Collection;
import java.util.Set;

import javax.annotation.Nullable;

import com.atlassian.crowd.embedded.api.PasswordConstraint;
import com.atlassian.crowd.embedded.api.PasswordCredential;
import com.atlassian.crowd.embedded.api.User;
import com.atlassian.crowd.embedded.api.ValidatePasswordRequest;
import com.atlassian.crowd.exception.InvalidCredentialException;
import com.atlassian.crowd.model.DirectoryEntity;
import com.atlassian.crowd.model.group.Group;
import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;

public class InternalDirectoryUtilsImpl implements InternalDirectoryUtils {

    public void validateDirectoryForEntity(final DirectoryEntity entity, Long directoryId) {
        Validate.notNull(entity, "entity cannot be null");
        Validate.notNull(entity.getDirectoryId(), "directoryId of entity cannot be null");
        Validate.isTrue(entity.getDirectoryId() == directoryId, "directoryId does not match the directoryId of the InternalDirectory");
    }

    public void validateUsername(String username) {
        if (StringUtils.isBlank(username)) {
            throw new IllegalArgumentException("A username must not be null or empty or blank");
        }
    }

    public void validateCredential(User user, PasswordCredential credential, final Set<PasswordConstraint> passwordConstraints,
                                   @Nullable String passwordComplexityMessage) throws InvalidCredentialException {
        Preconditions.checkNotNull(user);
        Preconditions.checkNotNull(credential);

        if (!credential.isEncryptedCredential()) {
            if (StringUtils.isBlank(credential.getCredential())) {
                throw new InvalidCredentialException("You cannot have an empty password");
            }

            Collection<PasswordConstraint> failedConstraints = ImmutableSet.copyOf(Iterables.filter(passwordConstraints, validationFailed(user, credential)));
            if (!failedConstraints.isEmpty()) {
                throw new InvalidCredentialException("The new password does not meet the directory complexity requirements",
                        StringUtils.stripToNull(passwordComplexityMessage), failedConstraints);
            }
        }
    }

    public void validateGroupName(Group group, String groupName) {
        if (StringUtils.isBlank(groupName)) {
            throw new IllegalArgumentException("A group name must not be null or empty or blank");
        }
    }

    private static Predicate<PasswordConstraint> validationFailed(final User user, final PasswordCredential password) {
        return new Predicate<PasswordConstraint>() {
            @Override
            public boolean apply(PasswordConstraint constraint) {
                return !constraint.validate(new ValidatePasswordRequest(password, user));
            }
        };
    }
}
