package com.atlassian.crowd.integration.rest.entity;

import com.atlassian.crowd.embedded.api.User;
import com.atlassian.crowd.embedded.api.UserComparator;
import com.atlassian.crowd.model.user.TimestampedUser;
import com.atlassian.crowd.model.user.UserWithAttributes;
import com.google.common.collect.ImmutableList;
import org.apache.commons.lang3.builder.ToStringBuilder;

import javax.annotation.Nullable;
import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlAttribute;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlRootElement;
import java.util.Collections;
import java.util.Date;
import java.util.Set;

/**
 * Represents a User entity (client side).
 *
 * @since v2.1
 */
@XmlRootElement(name = "user")
@XmlAccessorType(XmlAccessType.FIELD)
public class UserEntity implements UserWithAttributes, TimestampedUser {
    @XmlAttribute(name = "name")
    private final String name;

    @XmlElement(name = "first-name")
    private final String firstName;

    @XmlElement(name = "last-name")
    private final String lastName;

    @XmlElement(name = "display-name")
    private final String displayName;

    @XmlElement(name = "email")
    private final String emailAddress;

    @XmlElement(name = "password")
    private final PasswordEntity password; // will never be populated for a GET; only read for user creation or modification

    @XmlElement(name = "encrypted-password")
    private final PasswordEntity encryptedPassword; // will never be populated for a GET; only read for user creation or modification

    @XmlElement(name = "active")
    private final boolean active;

    @XmlElement(name = "created-date")
    private final Date createdDate;

    @XmlElement(name = "updated-date")
    private final Date updatedDate;

    @XmlElement(name = "key")
    private final String key;

    @XmlElement(name = "attributes")
    @Nullable
    private MultiValuedAttributeEntityList attributes;

    /**
     * JAXB requires a no-arg constructor.
     */
    private UserEntity() {
        this(null, null, null, null, null, null, false, null, null, null);
    }

    public UserEntity(final String name, final String firstName, final String lastName, final String displayName, final String emailAddress, final PasswordEntity password, final boolean active, final String key, final Date createdDate, final Date updatedDate) {
        this(name, firstName, lastName, displayName, emailAddress, password, active, key, createdDate, updatedDate, false);
    }

    public UserEntity(final String name, final String firstName, final String lastName, final String displayName, final String emailAddress, final PasswordEntity password, final boolean active, final String key, final Date createdDate, final Date updatedDate, boolean isPasswordEncrypted) {
        this.name = name;
        this.firstName = firstName;
        this.lastName = lastName;
        this.displayName = displayName;
        this.emailAddress = emailAddress;
        if (isPasswordEncrypted) {
            this.password = null;
            this.encryptedPassword = password;
        } else {
            this.password = password;
            this.encryptedPassword = null;
        }
        this.active = active;
        this.key = key;
        this.createdDate = createdDate;
        this.updatedDate = updatedDate;
        this.attributes = new MultiValuedAttributeEntityList(ImmutableList.<MultiValuedAttributeEntity>of());
    }

    public UserEntity(final String name, final String firstName, final String lastName, final String displayName, final String emailAddress, final PasswordEntity password, final boolean active) {
        this(name, firstName, lastName, displayName, emailAddress, password, active, null, null, null);
    }

    public UserEntity(final String name, final String firstName, final String lastName, final String displayName, final String emailAddress, final PasswordEntity password, final boolean active, boolean isPasswordEncrypted) {
        this(name, firstName, lastName, displayName, emailAddress, password, active, null, null, null, isPasswordEncrypted);
    }

    @Override
    public String getDisplayName() {
        return displayName;
    }

    @Override
    public String getFirstName() {
        return firstName;
    }

    @Override
    public String getLastName() {
        return lastName;
    }

    @Override
    public String getEmailAddress() {
        return emailAddress;
    }

    public PasswordEntity getPassword() {
        return password;
    }

    public PasswordEntity getEncryptedPassword() {
        return encryptedPassword;
    }

    @Override
    public boolean isActive() {
        return active;
    }

    @Override
    public String getName() {
        return name;
    }

    @Nullable
    @Override
    public Date getCreatedDate() {
        return createdDate;
    }

    @Nullable
    @Override
    public Date getUpdatedDate() {
        return updatedDate;
    }

    public void setAttributes(final MultiValuedAttributeEntityList attributes) {
        this.attributes = attributes;
    }

    public MultiValuedAttributeEntityList getAttributes() {
        return attributes;
    }

    @Override
    public Set<String> getValues(final String key) {
        return attributes != null ? attributes.getValues(key) : null;
    }

    @Override
    public String getValue(final String key) {
        return attributes != null ? attributes.getValue(key) : null;
    }

    @Override
    public Set<String> getKeys() {
        return attributes != null ? attributes.getKeys() : Collections.<String>emptySet();
    }

    @Override
    public boolean isEmpty() {
        return attributes == null || attributes.isEmpty();
    }

    /**
     * @return always <code>0</code>
     * @deprecated this method has never returned the directoryId and its existence violates the directory
     * encapsulation. It will be removed in future versions.
     */
    @Deprecated
    @Override
    public long getDirectoryId() {
        return 0L;
    }

    @Override
    public int compareTo(final User user) {
        return UserComparator.compareTo(this, user);
    }

    /**
     * The user key at the server is the externalId at the client.
     */
    @Override
    public String getExternalId() {
        return key;
    }

    @Override
    public boolean equals(final Object o) {
        return UserComparator.equalsObject(this, o);
    }

    @Override
    public int hashCode() {
        return UserComparator.hashCode(this);
    }

    @Override
    public String toString() {
        return new ToStringBuilder(this).
                append("name", getName()).
                append("active", isActive()).
                append("emailAddress", getEmailAddress()).
                append("firstName", getFirstName()).
                append("lastName", getLastName()).
                append("displayName", getDisplayName()).
                append("externalId", getExternalId()).
                append("attributes", getAttributes()).
                append("createdDate", getCreatedDate()).
                append("updatedDate", getUpdatedDate()).
                toString();
    }

    /**
     * Creates a new minimal user instance.
     *
     * @param username username for the user
     * @return minimal user instance
     */
    public static UserEntity newMinimalInstance(String username) {
        return new UserEntity(username, null, null, null, null, null, false, null, null, null);
    }
}
