package com.atlassian.crowd.integration.rest.service.factory;

import java.util.Properties;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.atlassian.crowd.integration.rest.service.BasicAuthRestExecutor;
import com.atlassian.crowd.integration.rest.service.DefaultHttpClientProvider;
import com.atlassian.crowd.integration.rest.service.HttpClientProvider;
import com.atlassian.crowd.integration.rest.service.RestCrowdClient;
import com.atlassian.crowd.model.authentication.ApplicationAuthenticationContext;
import com.atlassian.crowd.service.client.AuthenticationMethod;
import com.atlassian.crowd.service.client.ClientProperties;
import com.atlassian.crowd.service.client.ClientPropertiesImpl;
import com.atlassian.crowd.service.client.CrowdClient;
import com.atlassian.crowd.service.factory.CrowdClientFactory;

import org.apache.commons.lang3.StringUtils;
import org.apache.http.impl.client.CloseableHttpClient;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Factory class for creating a new instance of CrowdClient using REST.
 */
public class RestCrowdClientFactory implements CrowdClientFactory {
    private final HttpClientProvider httpClientProvider;

    public RestCrowdClientFactory() {
        this(new DefaultHttpClientProvider());
    }

    public RestCrowdClientFactory(HttpClientProvider httpClientProvider) {
        this.httpClientProvider = checkNotNull(httpClientProvider);
    }

    @Override
    public CrowdClient newInstance(final String url, final String applicationName, final String applicationPassword) {
        final ClientProperties clientProperties = new RestClientProperties(url, applicationName, applicationPassword);

        return newInstance(clientProperties);
    }

    @Override
    public CrowdClient newInstance(final ClientProperties clientProperties) {
        switch (clientProperties.getAuthenticationMethod()) {
            case BASIC_AUTH:
                return new RestCrowdClient(BasicAuthRestExecutor.createFrom(clientProperties, getHttpClient(clientProperties)));
            default:
                throw new IllegalArgumentException("Unknown authentication method '" + clientProperties.getAuthenticationMethod() + "'");
        }
    }

    /**
     * Get a {@link CloseableHttpClient} instance configured with the given {@link ClientProperties}.
     *
     * @param clientProperties the properties to use to configure the http client.
     * @return a {@link CloseableHttpClient} instance.
     */
    protected CloseableHttpClient getHttpClient(ClientProperties clientProperties) {
        return httpClientProvider.getClient(clientProperties);
    }

    /**
     * This class is used for forcing the use of specified url, application
     * name and application password. Values in ClientPropertiesImpl can
     * be overridden using system properties.
     */
    private static class RestClientProperties implements ClientProperties {
        private final ClientProperties delegate;
        private final String baseURL;
        private final String applicationName;
        private final String applicationPassword;

        RestClientProperties(final String url, final String applicationName, final String applicationPassword) {
            this.baseURL = StringUtils.removeEnd(checkNotNull(url), "/");
            this.applicationName = checkNotNull(applicationName);
            this.applicationPassword = checkNotNull(applicationPassword);
            this.delegate = ClientPropertiesImpl.newInstanceFromProperties(new Properties());
        }

        @Override
        public String getBaseURL() {
            return baseURL;
        }

        @Override
        public String getApplicationName() {
            return applicationName;
        }

        @Override
        public String getApplicationPassword() {
            return applicationPassword;
        }

        @Override
        public String getSSOCookieDomainName() {
            return delegate.getSSOCookieDomainName();
        }

        @Override
        @Nonnull
        public AuthenticationMethod getAuthenticationMethod() {
            return delegate.getAuthenticationMethod();
        }

        @Override
        public String getApplicationAuthenticationURL() {
            return delegate.getApplicationAuthenticationURL();
        }

        @Override
        public String getCookieTokenKey() {
            return delegate.getCookieTokenKey();
        }

        @Override
        public String getCookieTokenKey(String defaultKey) {
            return delegate.getCookieTokenKey(defaultKey);
        }

        @Override
        public String getSessionTokenKey() {
            return delegate.getSessionTokenKey();
        }

        @Override
        public String getSessionLastValidation() {
            return delegate.getSessionLastValidation();
        }

        @Override
        public long getSessionValidationInterval() {
            return delegate.getSessionValidationInterval();
        }

        @Override
        public ApplicationAuthenticationContext getApplicationAuthenticationContext() {
            return delegate.getApplicationAuthenticationContext();
        }

        @Override
        public String getHttpProxyPort() {
            return delegate.getHttpProxyPort();
        }

        @Override
        public String getHttpProxyHost() {
            return delegate.getHttpProxyHost();
        }

        @Override
        public String getHttpProxyUsername() {
            return delegate.getHttpProxyUsername();
        }

        @Override
        public String getHttpProxyPassword() {
            return delegate.getHttpProxyPassword();
        }

        @Override
        public String getHttpMaxConnections() {
            return delegate.getHttpMaxConnections();
        }

        @Override
        public String getHttpTimeout() {
            return delegate.getHttpTimeout();
        }

        @Override
        public String getSocketTimeout() {
            return delegate.getSocketTimeout();
        }

        @Override
        public void updateProperties(Properties properties) {
            delegate.updateProperties(properties);
        }
    }
}
