package com.atlassian.crowd.model.user;

import java.util.Date;

public final class ImmutableTimestampedUser extends BaseImmutableUser implements TimestampedUser {
    private final Date createdDate;
    private final Date updatedDate;

    private ImmutableTimestampedUser(Builder builder) {
        super(builder);
        this.createdDate = builder.createdDate;
        this.updatedDate = builder.updatedDate;
    }

    public static ImmutableTimestampedUser from(TimestampedUser user) {
        if (user instanceof ImmutableTimestampedUser) {
            return (ImmutableTimestampedUser) user;
        }
        return builder(user).build();
    }

    @Override
    public ImmutableTimestampedUser withName(String name) {
        return builder(this).name(name).build();
    }

    public static Builder builder(TimestampedUser user) {
        return new Builder(user);
    }

    public static Builder builder(String name) {
        return new Builder(name);
    }

    @Override
    public Date getCreatedDate() {
        return createdDate == null ? null : new Date(createdDate.getTime());
    }

    @Override
    public Date getUpdatedDate() {
        return updatedDate == null ? null : new Date(updatedDate.getTime());
    }

    public static class Builder extends BaseImmutableUser.Builder<Builder> {
        private Date createdDate;
        private Date updatedDate;

        private Builder(TimestampedUser user) {
            super(user);
            createdDate(user.getCreatedDate());
            updatedDate(user.getUpdatedDate());
        }

        private Builder(String name) {
            super(name);
        }

        public Builder createdDate(Date createdDate) {
            this.createdDate = createdDate;
            return this;
        }

        public Builder updatedDate(Date updatedDate) {
            this.updatedDate = updatedDate;
            return this;
        }

        @Override
        public ImmutableTimestampedUser build() {
            return new ImmutableTimestampedUser(this);
        }
    }
}
