package com.atlassian.crowd.search.query.entity;

import com.atlassian.crowd.embedded.api.Query;
import com.atlassian.crowd.embedded.api.SearchRestriction;
import com.atlassian.crowd.search.EntityDescriptor;
import com.atlassian.crowd.search.builder.Combine;
import com.atlassian.crowd.search.builder.QueryBuilder;
import com.atlassian.crowd.search.query.entity.restriction.BooleanRestriction;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.builder.ToStringBuilder;

import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

public abstract class EntityQuery<T> implements Query<T> {
    private final EntityDescriptor entityDescriptor;
    private final SearchRestriction searchRestriction;
    private final int startIndex;
    private final int maxResults;
    private Class<T> returnType;
    /**
     * This is the recommended maximum number of 'max' results the system will allow you to return. This value is
     * <strong>NOT</strong> enforced. ApplicationServiceGeneric often retrieves (startIndex + maxResults) number of
     * results which breaks this MAX_MAX_RESULTS.
     */
    public static final int MAX_MAX_RESULTS = 1000;

    /**
     * Flag to indicate that an EntityQuery should retrieve all results.
     * <p>
     * <strong>WARNING:</strong> using this flag could retrieve thousands or millions of entities. Misuse can cause
     * <em>massive performance problems</em>. This flag should only ever be used in exceptional circumstances.
     * <p>
     * If you need to find "all" entities, then consider making multiple successive calls to Crowd to receive
     * partial results. That way, the entire result set is never stored in memory on the Crowd server at
     * any one time.
     */
    public static final int ALL_RESULTS = -1;

    public EntityQuery(final Class<T> returnType, final EntityDescriptor entityDescriptor, final SearchRestriction searchRestriction, final int startIndex, final int maxResults) {
        Validate.notNull(entityDescriptor, "entity cannot be null");
        Validate.notNull(searchRestriction, "searchRestriction cannot be null");
        Validate.notNull(returnType, "returnType cannot be null");
        Validate.isTrue(maxResults == ALL_RESULTS || maxResults > 0, "maxResults must be greater than 0 (unless set to EntityQuery.ALL_RESULTS)");
        Validate.isTrue(startIndex >= 0, "startIndex cannot be less than zero");

        this.entityDescriptor = entityDescriptor;
        this.searchRestriction = searchRestriction;
        this.startIndex = startIndex;
        this.maxResults = maxResults;
        this.returnType = returnType;
    }

    public EntityQuery(final EntityQuery query, final Class<T> returnType) {
        this(returnType, query.getEntityDescriptor(), query.getSearchRestriction(), query.getStartIndex(), query.getMaxResults());
    }

    public EntityQuery(final EntityQuery<T> query, final int startIndex, final int maxResults) {
        this(query.getReturnType(), query.getEntityDescriptor(), query.getSearchRestriction(), startIndex, maxResults);
    }

    public EntityDescriptor getEntityDescriptor() {
        return entityDescriptor;
    }

    public SearchRestriction getSearchRestriction() {
        return searchRestriction;
    }

    public int getStartIndex() {
        return startIndex;
    }

    public int getMaxResults() {
        return maxResults;
    }

    public Class<T> getReturnType() {
        return returnType;
    }

    @Override
    public boolean equals(final Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof EntityQuery)) {
            return false;
        }

        EntityQuery query = (EntityQuery) o;

        if (maxResults != query.maxResults) {
            return false;
        }
        if (startIndex != query.startIndex) {
            return false;
        }
        if (entityDescriptor != null ? !entityDescriptor.equals(query.entityDescriptor) : query.entityDescriptor != null) {
            return false;
        }
        if (returnType != query.returnType) {
            return false;
        }
        //noinspection RedundantIfStatement
        if (searchRestriction != null ? !searchRestriction.equals(query.searchRestriction) : query.searchRestriction != null) {
            return false;
        }

        return true;
    }

    /**
     * Simple utility method that increases or decreases the result limit and deals with {@link #ALL_RESULTS},
     * overflow and prevents negative values.
     */
    public static int addToMaxResults(int maxResults, int add) {
        if (maxResults == ALL_RESULTS) {
            return ALL_RESULTS;
        }
        long sum = (long) maxResults + add;
        if (sum < 0) {
            return 0;
        } else if (sum > Integer.MAX_VALUE) {
            return Integer.MAX_VALUE;
        } else {
            return (int) sum;
        }
    }

    /**
     * Converts max results limit to long value - replaces {@link #ALL_RESULTS} with {@link Long#MAX_VALUE}}.
     */
    public static long allResultsToLongMax(int maxResults) {
        return maxResults == ALL_RESULTS ? Long.MAX_VALUE : maxResults;
    }

    @Override
    public int hashCode() {
        int result = entityDescriptor != null ? entityDescriptor.hashCode() : 0;
        result = 31 * result + (searchRestriction != null ? searchRestriction.hashCode() : 0);
        result = 31 * result + startIndex;
        result = 31 * result + maxResults;
        result = 31 * result + (returnType != null ? returnType.hashCode() : 0);
        return result;
    }

    @Override
    public String toString() {
        return new ToStringBuilder(this).
                append("entity", entityDescriptor).
                append("returnType", returnType).
                append("searchRestriction", searchRestriction).
                append("startIndex", startIndex).
                append("maxResults", maxResults).
                toString();
    }

    public <Q> EntityQuery<Q> withReturnType(Class<Q> returnType) {
        return QueryBuilder.queryFor(returnType, entityDescriptor, searchRestriction, startIndex, maxResults);
    }

    public EntityQuery<T> withStartIndex(int startIndex) {
        return withStartIndexAndMaxResults(startIndex, maxResults);
    }

    public EntityQuery<T> withMaxResults(int maxResults) {
        return withStartIndexAndMaxResults(startIndex, maxResults);
    }

    public EntityQuery<T> withStartIndexAndMaxResults(int startIndex, int maxResults) {
        return QueryBuilder.queryFor(returnType, entityDescriptor, searchRestriction, startIndex, maxResults);
    }

    public EntityQuery<T> withSearchRestriction(SearchRestriction searchRestriction) {
        return QueryBuilder.queryFor(returnType, entityDescriptor, searchRestriction, startIndex, maxResults);
    }

    /**
     * @return query with adjusted start index and maximum results which is required for merging query results
     */
    public EntityQuery<T> baseSplitQuery() {
        return withStartIndexAndMaxResults(0, addToMaxResults(this.maxResults, startIndex));
    }

    /**
     * Splits the query if needed. Query is split if the number of OR conditions is higher than {@code maxSize}.
     * Only top level condition split is supported. Note that the start index and max conditions are adjusted.
     * @return split queries if split is needed or {@code Optional#empty} otherwise
     */
    public Optional<List<EntityQuery<T>>> splitOrRestrictionIfNeeded(int maxSize) {
        if (searchRestriction instanceof BooleanRestriction) {
            BooleanRestriction booleanRestriction = (BooleanRestriction) searchRestriction;
            if (booleanRestriction.getBooleanLogic() == BooleanRestriction.BooleanLogic.OR
                    && booleanRestriction.getRestrictions().size() > maxSize) {
                EntityQuery<T> base = baseSplitQuery();
                return Optional.of(
                        Lists.partition(ImmutableList.copyOf(booleanRestriction.getRestrictions()), maxSize).stream()
                                .map(restrictions -> base.withSearchRestriction(Combine.anyOf(restrictions)))
                                .collect(Collectors.toList()));
            }
        }
        return Optional.empty();
    }

    public EntityQuery<T> withAllResults() {
        return withStartIndexAndMaxResults(0, ALL_RESULTS);
    }

    public EntityQuery<T> addToMaxResults(int add) {
        return withMaxResults(addToMaxResults(this.maxResults, add));
    }

    public boolean hasAllResults() {
        return startIndex == 0 && maxResults == ALL_RESULTS;
    }
}
