package com.atlassian.crowd.model.application;

import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.embedded.api.OperationType;
import com.atlassian.crowd.model.directory.ImmutableDirectory;
import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableSet;

import java.util.Objects;
import java.util.Set;

/**
 * An immutable representation of a {@link ApplicationDirectoryMapping}
 *
 * @since v2.12
 */
public class ImmutableApplicationDirectoryMapping implements ApplicationDirectoryMapping {
    private final ImmutableDirectory directory;
    private final boolean allowAllToAuthenticate;
    private final ImmutableSet<String> authorisedGroupNames;
    private final ImmutableSet<OperationType> allowedOperations;

    private ImmutableApplicationDirectoryMapping(ImmutableApplicationDirectoryMapping.Builder builder) {
        this.directory = builder.directory;
        this.allowAllToAuthenticate = builder.allowAllToAuthenticate;
        this.authorisedGroupNames = builder.authorisedGroupNames;
        this.allowedOperations = builder.allowedOperations;
    }

    public static ImmutableApplicationDirectoryMapping.Builder builder() {
        return new Builder();
    }

    public static ImmutableApplicationDirectoryMapping.Builder builder(ApplicationDirectoryMapping applicationDirectoryMapping) {
        return new Builder(Preconditions.checkNotNull(applicationDirectoryMapping));
    }

    public static ImmutableApplicationDirectoryMapping from(ApplicationDirectoryMapping applicationDirectoryMapping) {
        return applicationDirectoryMapping instanceof ImmutableApplicationDirectoryMapping
                ? (ImmutableApplicationDirectoryMapping) applicationDirectoryMapping
                : builder(applicationDirectoryMapping).build();
    }

    @Override
    public Directory getDirectory() {
        return directory;
    }

    @Override
    public boolean isAllowAllToAuthenticate() {
        return allowAllToAuthenticate;
    }

    @Override
    public Set<String> getAuthorisedGroupNames() {
        return authorisedGroupNames;
    }

    @Override
    public Set<OperationType> getAllowedOperations() {
        return allowedOperations;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        ImmutableApplicationDirectoryMapping that = (ImmutableApplicationDirectoryMapping) o;
        return allowAllToAuthenticate == that.allowAllToAuthenticate &&
                Objects.equals(directory, that.directory) &&
                Objects.equals(authorisedGroupNames, that.authorisedGroupNames) &&
                Objects.equals(allowedOperations, that.allowedOperations);
    }

    @Override
    public int hashCode() {
        return Objects.hash(directory, allowAllToAuthenticate, authorisedGroupNames, allowedOperations);
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("directory", directory)
                .add("allowAllToAuthenticate", allowAllToAuthenticate)
                .add("authorisedGroupNames", authorisedGroupNames)
                .add("allowedOperations", allowedOperations)
                .toString();
    }

    public static final class Builder {
        private ImmutableDirectory directory;
        private boolean allowAllToAuthenticate;
        private ImmutableSet<String> authorisedGroupNames = ImmutableSet.of();
        private ImmutableSet<OperationType> allowedOperations = ImmutableSet.of();

        private Builder() {
        }

        private Builder(ApplicationDirectoryMapping mapping) {
            setDirectory(mapping.getDirectory());
            setAllowedOperations(mapping.getAllowedOperations());
            setAuthorisedGroupNames(mapping.getAuthorisedGroupNames());
            setAllowAllToAuthenticate(mapping.isAllowAllToAuthenticate());
        }

        public Builder setDirectory(Directory directory) {
            this.directory = ImmutableDirectory.from(directory);
            return this;
        }

        public Builder setAllowAllToAuthenticate(boolean allowAllToAuthenticate) {
            this.allowAllToAuthenticate = allowAllToAuthenticate;
            return this;
        }

        public Builder setAuthorisedGroupNames(Set<String> authorisedGroupNames) {
            this.authorisedGroupNames = ImmutableSet.copyOf(authorisedGroupNames);
            return this;
        }

        public Builder setAllowedOperations(Set<OperationType> allowedOperations) {
            this.allowedOperations = ImmutableSet.copyOf(allowedOperations);
            return this;
        }

        public ImmutableApplicationDirectoryMapping build() {
            return new ImmutableApplicationDirectoryMapping(this);
        }
    }
}