package com.atlassian.crowd.model.application;

import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.embedded.api.OperationType;
import com.atlassian.crowd.embedded.api.PasswordCredential;
import com.atlassian.crowd.model.InternalEntity;
import com.atlassian.crowd.model.InternalEntityTemplate;
import com.atlassian.crowd.model.webhook.Webhook;
import com.atlassian.crowd.util.InternalEntityUtils;
import com.google.common.base.Functions;
import com.google.common.base.Strings;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.builder.ToStringBuilder;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import static com.atlassian.crowd.embedded.impl.IdentifierUtils.toLowerCase;

/**
 * Implementation of Application (designed for use for Hibernate).
 *
 * @deprecated Use {@link Application} or {@link ImmutableApplication} instead. Since v2.12.
 */
@Deprecated
public class ApplicationImpl extends InternalEntity implements Application {
    private String lowerName;
    private ApplicationType type;
    private String description;
    private PasswordCredential credential;

    private Map<String, String> attributes = new HashMap<String, String>();
    private List<DirectoryMapping> directoryMappings = new ArrayList<DirectoryMapping>();
    private Set<RemoteAddress> remoteAddresses = new HashSet<RemoteAddress>();
    private Set<Webhook> webhooks = new HashSet<Webhook>();

    protected ApplicationImpl() {
    }

    protected ApplicationImpl(final String name, final long id, final ApplicationType type, final String description,
                              final PasswordCredential credential, final boolean active, final Map<String, String> attributes,
                              final List<DirectoryMapping> directoryMappings, final Set<RemoteAddress> remoteAddresses,
                              final Set<Webhook> webhooks,
                              final Date createdDate, final Date updatedDate) {
        setNameAndLowerName(name);
        this.id = id;
        this.type = type;
        this.description = description;
        this.credential = credential;
        this.active = active;
        this.attributes = MapUtils.isEmpty(attributes) ? Maps.<String, String>newHashMap() : Maps.newHashMap(attributes);
        this.directoryMappings = CollectionUtils.isEmpty(directoryMappings) ? Lists.<DirectoryMapping>newArrayList() : Lists.newArrayList(
                directoryMappings);
        this.remoteAddresses = CollectionUtils.isEmpty(remoteAddresses) ? Sets.<RemoteAddress>newHashSet() : Sets.newHashSet(remoteAddresses);
        this.webhooks = CollectionUtils.isEmpty(webhooks) ? Sets.<Webhook>newHashSet() : Sets.newHashSet(webhooks);
        this.createdDate = (createdDate == null) ? new Date() : new Date(createdDate.getTime());
        this.updatedDate = (updatedDate == null) ? new Date() : new Date(updatedDate.getTime());
    }

    public static ApplicationImpl convertIfNeeded(final Application application) {
        return (application instanceof ApplicationImpl) ? (ApplicationImpl) application : newInstance(application);
    }

    public static ApplicationImpl newInstance(final Application application) {
        final long applicationId = (application.getId() == null ? -1L : application.getId());
        return new ApplicationImpl(application.getName(), applicationId, application.getType(),
                application.getDescription(), application.getCredential(), application.isActive(),
                application.getAttributes(), application.getDirectoryMappings(), application.getRemoteAddresses(),
                application.getWebhooks(),
                application.getCreatedDate(), application.getUpdatedDate());
    }

    public static ApplicationImpl newInstance(final String name, final ApplicationType type) {
        return newInstanceWithPassword(name, type, null);
    }

    public static ApplicationImpl newInstanceWithIdAndCredential(final String name, final ApplicationType type, final PasswordCredential credential, final long id) {
        return new ApplicationImpl(name, id, type, null, credential, true, null, null, null, null, null, null);
    }

    public static ApplicationImpl newInstanceWithCredential(final String name, final ApplicationType type, final PasswordCredential credential) {
        return new ApplicationImpl(name, -1L, type, null, credential, true, null, null, null, null, null, null);
    }

    public static ApplicationImpl newInstanceWithPassword(final String name, final ApplicationType type, final String password) {
        return newInstanceWithCredential(name, type, PasswordCredential.unencrypted(password));
    }

    /**
     * Used for importing via XML migration.
     *
     * @param template directory template.
     */
    public ApplicationImpl(InternalEntityTemplate template) {
        super(template);
    }

    /**
     * Only to be used by the ApplicationDAO#update method
     */
    public void updateDetailsFromApplication(Application application) {
        setName(application.getName());
        setDescription(application.getDescription());
        setType(application.getType());
        setActive(application.isActive());

        updateAttributesFrom(application.getAttributes());
        setAliasingEnabled(application.isAliasingEnabled());
        setMembershipAggregationEnabled(application.isMembershipAggregationEnabled());
        setAuthenticationOrderOptimizationForCachedDirectoriesEnabled(application.isCachedDirectoriesAuthenticationOrderOptimisationEnabled());
        setLowerCaseOutput(application.isLowerCaseOutput());

        setRemoteAddresses(application.getRemoteAddresses());

        // because this method is only called from ApplicationDAO#update, 'this' is a persistent instance,
        // and we cannot modify the instance of the 'webhooks' collection because the relationship is
        // owned by WebhookImpl
        webhooks.retainAll(application.getWebhooks());
        webhooks.addAll(application.getWebhooks());
    }

    public void updateAttributesFrom(Map<String, String> attributes) {
        // Avoid discarding original collection, so that Hibernate won't perform one shot delete.
        this.attributes.entrySet().retainAll(attributes.entrySet());
        this.attributes.putAll(Maps.filterValues(attributes, value -> !Strings.isNullOrEmpty(value)));
    }

    public void validate() {
        Validate.notNull(name, "name cannot be null");
        Validate.isTrue(toLowerCase(name).equals(lowerName), "lowerName must be the lower-case representation of name");
        Validate.notNull(type, "type cannot be null");
        Validate.notNull(credential, "credential cannot be null");
        Validate.notNull(credential.getCredential(), "credential cannot have null value");
        Validate.isTrue(credential.isEncryptedCredential(), "credential must be encrypted");
        Validate.notNull(createdDate, "createdDate cannot be null");
        Validate.notNull(updatedDate, "updatedDate cannot be null");
        Validate.notNull(attributes, "attributes cannot be null");
        Validate.notNull(directoryMappings, "directoryMappings cannot be null");
        Validate.notNull(remoteAddresses, "remoteAddresses cannot be null");
        Validate.notNull(webhooks, "webhooks cannot be null");
    }

    @Override
    public void setName(final String name) {
        setNameAndLowerName(name);
    }

    @Override
    public void setActive(final boolean active) {
        this.active = active;
    }

    public String getLowerName() {
        return lowerName;
    }

    private void setLowerName(final String lowerName) {
        this.lowerName = lowerName;
    }

    @Override
    public ApplicationType getType() {
        return type;
    }

    public void setType(final ApplicationType type) {
        Validate.notNull(type);
        this.type = type;
    }

    @Override
    public String getDescription() {
        return description;
    }

    public void setDescription(final String description) {
        this.description = description;
    }

    @Override
    public PasswordCredential getCredential() {
        return credential;
    }

    public void setCredential(final PasswordCredential credential) {
        this.credential = credential;
    }

    @Override
    public boolean isPermanent() {
        return type.equals(ApplicationType.CROWD)
                // the current implementation of crowd-application plugins is tightly coupled with the application name - prevent changing it
                || type.equals(ApplicationType.PLUGIN);

    }

    @Override
    public Map<String, String> getAttributes() {
        return attributes;
    }

    /**
     * Sets the attributes of the application. <code>attributes</code> must be a mutable <tt>Map</tt>.
     *
     * @param attributes new attributes
     */
    public void setAttributes(final Map<String, String> attributes) {
        this.attributes = attributes;
    }

    @Override
    public List<DirectoryMapping> getDirectoryMappings() {
        return directoryMappings;
    }

    @Nonnull
    @Override
    public List<ApplicationDirectoryMapping> getApplicationDirectoryMappings() {
        return Lists.transform(directoryMappings, Functions.identity());
    }

    public void addDirectoryMapping(final Directory directory, final boolean allowAllToAuthenticate, final OperationType... operationTypes) {
        DirectoryMapping directoryMapping = getDirectoryMapping(directory.getId());
        if (directoryMapping == null) {
            directoryMapping = new DirectoryMapping(this, directory, allowAllToAuthenticate, new HashSet<OperationType>(Arrays.asList(operationTypes)));
            directoryMappings.add(directoryMapping);
        } else {
            directoryMapping.setAllowAllToAuthenticate(allowAllToAuthenticate);
            directoryMapping.setAllowedOperations(new HashSet<OperationType>(Arrays.asList(operationTypes)));
        }
    }

    public void addGroupMapping(final long directoryId, final String groupName) {
        DirectoryMapping directoryMapping = getDirectoryMapping(directoryId);

        if (directoryMapping == null) {
            throw new IllegalArgumentException("The application <" + name + "> does not contain a directory mapping for directory with id <" + directoryId + ">");
        }

        directoryMapping.addGroupMapping(groupName);
    }

    //TODO, fix up the DirectoryMapping so it can't be mutated
    @Override
    public DirectoryMapping getDirectoryMapping(long directoryId) {
        for (DirectoryMapping mapping : directoryMappings) {
            if (mapping.getDirectory().getId() == directoryId) {
                return mapping;
            }
        }

        return null;
    }

    @Nullable
    @Override
    public ApplicationDirectoryMapping getApplicationDirectoryMapping(long directoryId) {
        return getDirectoryMapping(directoryId);
    }

    public boolean removeDirectoryMapping(long directoryId) {
        DirectoryMapping mapping = getDirectoryMapping(directoryId);

        return directoryMappings.remove(mapping);
    }

    private void setDirectoryMappings(List<DirectoryMapping> directoryMappings) {
        this.directoryMappings = directoryMappings;
    }

    @Override
    public Set<RemoteAddress> getRemoteAddresses() {
        return remoteAddresses;
    }

    public void addRemoteAddress(String remoteAddress) {
        remoteAddresses.add(new RemoteAddress(remoteAddress));
    }

    public void setRemoteAddresses(final Set<RemoteAddress> remoteAddresses) {
        this.remoteAddresses = remoteAddresses;
    }

    @Override
    public boolean hasRemoteAddress(final String remoteAddress) {
        return getRemoteAddresses().contains(new RemoteAddress(remoteAddress));
    }

    public boolean removeRemoteAddress(String remoteAddress) {
        return getRemoteAddresses().remove(new RemoteAddress(remoteAddress));
    }

    @Override
    public Set<Webhook> getWebhooks() {
        return webhooks;
    }

    /**
     * This setter should only be called by hibernate and must replace the collection instance; compare with
     * {@link #updateDetailsFromApplication(Application)}.
     *
     * @param webhooks new Set of webhooks
     */
    public void setWebhooks(Set<Webhook> webhooks) {
        this.webhooks = webhooks;
    }

    /**
     * @param name attribute name.
     * @return a collection of the only attribtue value or <code>null</code>
     * if the directory does not have the attribute.
     */
    @Override
    public Set<String> getValues(final String name) {
        String value = getValue(name);
        if (value != null) {
            return Collections.singleton(value);
        } else {
            return null;
        }
    }

    @Override
    public String getValue(final String name) {
        return attributes.get(name);
    }

    @Override
    public Set<String> getKeys() {
        return attributes.keySet();
    }

    @Override
    public boolean isEmpty() {
        return attributes.isEmpty();
    }

    public void setAttribute(final String name, final String value) {
        attributes.put(name, value);
    }

    public void removeAttribute(final String name) {
        attributes.remove(name);
    }

    @Override
    public boolean isFilteringUsersWithAccessEnabled() {
        return getBooleanAttribute(ApplicationAttributeConstants.ATTRIBUTE_KEY_FILTER_USERS_WITH_ACCESS);
    }

    public void setFilterUsersWithAccessEnabled(final boolean filterUsersWithAccessEnabled) {
        setAttribute(ApplicationAttributeConstants.ATTRIBUTE_KEY_FILTER_USERS_WITH_ACCESS, Boolean.toString(filterUsersWithAccessEnabled));
    }

    @Override
    public boolean isFilteringGroupsWithAccessEnabled() {
        return getBooleanAttribute(ApplicationAttributeConstants.ATTRIBUTE_KEY_FILTER_GROUPS_WITH_ACCESS);
    }

    public void setFilterGroupsWithAccessEnabled(final boolean filterGroupsWithAccessEnabled) {
        setAttribute(ApplicationAttributeConstants.ATTRIBUTE_KEY_FILTER_GROUPS_WITH_ACCESS, Boolean.toString(filterGroupsWithAccessEnabled));
    }

    @Override
    public boolean isLowerCaseOutput() {
        return getBooleanAttribute(ApplicationAttributeConstants.ATTRIBUTE_KEY_LOWER_CASE_OUTPUT);
    }

    public void setLowerCaseOutput(boolean value) {
        setAttribute(ApplicationAttributeConstants.ATTRIBUTE_KEY_LOWER_CASE_OUTPUT, Boolean.toString(value));
    }

    public void setAliasingEnabled(final boolean aliasingEnabled) {
        setAttribute(ApplicationAttributeConstants.ATTRIBUTE_KEY_ALIASING_ENABLED, Boolean.toString(aliasingEnabled));
    }

    @Override
    public boolean isAliasingEnabled() {
        return getBooleanAttribute(ApplicationAttributeConstants.ATTRIBUTE_KEY_ALIASING_ENABLED);
    }

    @Override
    public boolean isMembershipAggregationEnabled() {
        return getBooleanAttribute(ApplicationAttributeConstants.ATTRIBUTE_KEY_AGGREGATE_MEMBERSHIPS);
    }

    public void setMembershipAggregationEnabled(final boolean membershipAggregationEnabled) {
        setAttribute(
                ApplicationAttributeConstants.ATTRIBUTE_KEY_AGGREGATE_MEMBERSHIPS,
                Boolean.toString(membershipAggregationEnabled));
    }

    @Override
    public boolean isCachedDirectoriesAuthenticationOrderOptimisationEnabled() {
        return getBooleanAttribute(ApplicationAttributeConstants.ATTRIBUTE_KEY_OPTIMISE_CACHED_DIRECTORIES_AUTHENTICATION_ORDER);
    }

    private boolean getBooleanAttribute(final String name) {
        return Boolean.parseBoolean(getValue(name));
    }

    public void setAuthenticationOrderOptimizationForCachedDirectoriesEnabled(final boolean authenticationOrderOptimizationForCachedDirectoriesEnabled) {
        setAttribute(
                ApplicationAttributeConstants.ATTRIBUTE_KEY_OPTIMISE_CACHED_DIRECTORIES_AUTHENTICATION_ORDER,
                Boolean.toString(authenticationOrderOptimizationForCachedDirectoriesEnabled));
    }

    @Override
    public boolean isAuthenticationWithoutPasswordEnabled() {
        return getBooleanAttribute(ApplicationAttributeConstants.AUTHENTICATION_WITHOUT_PASSWORD_ENABLED);
    }

    public void setAuthenticationWithoutPasswordEnabled(final boolean authenticationWithoutPasswordEnabled) {
        setAttribute(ApplicationAttributeConstants.AUTHENTICATION_WITHOUT_PASSWORD_ENABLED,
                Boolean.toString(authenticationWithoutPasswordEnabled));
    }

    @Override
    public boolean equals(final Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof ApplicationImpl)) {
            return false;
        }

        ApplicationImpl that = (ApplicationImpl) o;

        if (getLowerName() != null ? !getLowerName().equals(that.getLowerName()) : that.getLowerName() != null) {
            return false;
        }

        return true;
    }

    @Override
    public int hashCode() {
        return getLowerName() != null ? getLowerName().hashCode() : 0;
    }

    @Override
    public String toString() {
        return new ToStringBuilder(this).
                append("lowerName", getLowerName()).
                append("type", getType()).
                append("description", getDescription()).
                append("credential", getCredential()).
                append("attributes", getAttributes()).
                append("directoryMappings", getDirectoryMappings()).
                append("remoteAddresses", getRemoteAddresses()).
                append("webhooks", getWebhooks()).
                toString();
    }

    private void setNameAndLowerName(final String name) {
        Validate.notNull(name);
        InternalEntityUtils.validateLength(name);
        this.name = name;
        this.lowerName = toLowerCase(name);
    }
}
