package com.atlassian.crowd.manager.directory;

import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.embedded.api.DirectorySynchronisationInformation;
import com.atlassian.crowd.exception.DirectoryNotFoundException;
import com.atlassian.crowd.model.directory.SynchronisationStatusKey;

import javax.annotation.Nullable;
import java.io.Serializable;
import java.util.List;

public interface SynchronisationStatusManager {
    /**
     * Notify that directory synchronisation has started.
     *
     * @param directory directory
     */
    void syncStarted(Directory directory);

    /**
     * Notify that directory synchronisation status has changed.
     *
     * @param directoryId directory id
     * @param key         synchronisation status message key
     * @param parameters  synchronisation status message parameters
     * @throws IllegalStateException if the directory is not currently synchronising
     * @deprecated since 2.12.0, use {@link SynchronisationStatusManager#syncStatus(long, SynchronisationStatusKey, List)} instead
     */
    @Deprecated
    void syncStatus(long directoryId, String key, Serializable... parameters);

    /**
     * Notify that directory synchronisation status has changed.
     *
     * @param directoryId directory id
     * @param statusKey   synchronisation status message key
     * @param parameters  synchronisation status message parameters
     */
    void syncStatus(long directoryId, SynchronisationStatusKey statusKey, List<Serializable> parameters);

    /**
     * Notify about a failure that occurs during synchronisation.
     *
     * @param directoryId directory id
     * @param mode        synchronisation mode
     * @param throwable   throwable that carries information about the failure
     */
    default void syncFailure(long directoryId, SynchronisationMode mode, Throwable throwable) {
        throw new UnsupportedOperationException();
    }

    /**
     * Notify that directory synchronisation has finished.
     *
     * @param directoryId directory id
     * @deprecated since 2.12.0, use {@link SynchronisationStatusManager#syncFinished(long, SynchronisationStatusKey, List, String)} instead
     */
    @Deprecated
    void syncFinished(long directoryId);

    /**
     * Notify that directory synchronisation has finished.
     *
     * @param directoryId     directory id
     * @param statusKey       synchronisation status
     * @param parameters      parameters for statusKey
     */
    void syncFinished(long directoryId, SynchronisationStatusKey statusKey, List<Serializable> parameters);

    /**
     * Removes statuses for the given directory
     *
     * @param directoryId directory id
     */
    void removeStatusesForDirectory(long directoryId);

    /**
     * Returns directory synchronisation information. The returned value is never be null.
     *
     * @param directory directory to retrieve information from
     * @return directory synchronisation information
     * @deprecated since 2.12.0, use {@link SynchronisationStatusManager#getDirectorySynchronisationInformation(long)} instead
     */
    @Deprecated
    DirectorySynchronisationInformation getDirectorySynchronisationInformation(Directory directory);


    /**
     * Returns directory synchronisation information. The returned value is never be null.
     *
     * @param directoryId directory id to retrieve information from
     * @return directory synchronisation information
     */
    DirectorySynchronisationInformation getDirectorySynchronisationInformation(long directoryId) throws DirectoryNotFoundException;

    /**
     * Returns the last stored synchronisation token for the directory
     * @param directoryId the id of the directory
     * @return the last stored synchronisation token or null if there is no token stored in the database
     */
    @Nullable
    String getLastSynchronisationTokenForDirectory(long directoryId);

    /**
     * Store a new synchronisation token for the given directory. This will overwrite the old token.
     * @param directoryId the id of the directory
     * @param synchronisationToken the new token to store
     */
    void storeSynchronisationTokenForDirectory(long directoryId, String synchronisationToken);

    /**
     * Clears the synchronisation token for the given directory
     *
     * @param directoryId directory id
     */
    void clearSynchronisationTokenForDirectory(long directoryId);
}
