package com.atlassian.crowd.manager.audit;

import com.atlassian.crowd.audit.AuditLogChangeset;
import com.atlassian.crowd.audit.query.AuditLogQuery;

import java.util.List;

/**
 * The service for interacting with the audit log. It allows to create new audit log entries or query the audit log.
 *
 * @since v2.12
 */
public interface AuditService {
    /**
     * Persists an audit log changeset. The timestamp will be set to the current time.
     * Will not do anything when audit log is disabled.
     *
     * @param changeset the changeset that will be persisted
     */
    void saveAudit(AuditLogChangeset changeset);

    /**
     * Queries the audit log. Returns a {@link List} of {@link AuditLogChangeset}s, which match the criteria specified in
     * the query. The results will be sorted by the timestamp, descending.
     *
     * @since v3.2
     * @param query the search query
     * @return List of AuditLogEntry matching the query
     */
    <RESULT> List<RESULT> searchAuditLog(AuditLogQuery<RESULT> query);

    /**
     * @return true if the audit log is enabled, false otherwise. A disabled audit log will not persist any new entries.
     * @deprecated As of v3.2, replaced by {@link AuditService#shouldAuditEvent()}
     */
    @Deprecated
    boolean isEnabled();

    /**
     * Sets the new configuration for auditing
     *
     * @param auditLogConfiguration new configuration
     */
    void saveConfiguration(AuditLogConfiguration auditLogConfiguration);

    /**
     * Returns current auditing configuration
     *
     * @return current configuration
     */
    AuditLogConfiguration getConfiguration();

    /**
     * Allows the caller to determine whether it should attempt to generate audit events. Auditing might be disabled
     * globally, or it might be disabled during specific operations or states (for example during synchronizations).
     *
     * @since v3.2
     * @return true if calling saveAudit() will persist the event, false otherwise
     */
    boolean shouldAuditEvent();
}
