package com.atlassian.crowd.audit;

import com.google.common.base.MoreObjects;
import com.google.common.collect.ImmutableSet;

import javax.annotation.Nonnull;
import java.time.Instant;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * Concrete implementation of an audit log changeset
 */
public class ImmutableAuditLogChangeset implements AuditLogChangeset {
    private final Long id;
    private final Instant timestamp;
    private final AuditLogAuthor author;
    private final AuditLogEventType eventType;
    private final String ipAddress;
    private final String eventMessage;
    private final AuditLogEventSource source;
    private final Set<ImmutableAuditLogEntry> entries;
    private final Set<ImmutableAuditLogEntity> entities;

    /**
     * @deprecated Use the {@link ImmutableAuditLogChangeset.Builder} instead
     */
    @Deprecated
    public ImmutableAuditLogChangeset(
            Long id,
            Instant timestamp,
            AuditLogAuthorType authorType,
            Long authorId,
            String authorName,
            AuditLogEventType eventType,
            AuditLogEntityType entityType,
            Long entityId,
            String entityName,
            String ipAddress,
            String eventMessage,
            List<ImmutableAuditLogEntry> entries) {
        this.id = id;
        this.timestamp = timestamp;
        this.author = new ImmutableAuditLogAuthor(authorId,  authorName, authorType);
        this.eventType = eventType;
        this.ipAddress = ipAddress;
        this.eventMessage = eventMessage;
        this.source = AuditLogEventSource.MANUAL;
        this.entries = ImmutableSet.copyOf(entries);
        if (hasEntity(entityType, entityId, entityName)) {
            this.entities = Collections.singleton(new ImmutableAuditLogEntity.Builder()
                    .setEntityType(entityType)
                    .setEntityId(entityId)
                    .setEntityName(entityName)
                    .setPrimary()
                    .build()
            );
        } else {
            this.entities = Collections.emptySet();
        }
    }

    private ImmutableAuditLogChangeset(ImmutableAuditLogChangeset.Builder builder) {
        this.id = builder.id;
        this.timestamp = builder.timestamp;
        this.author = builder.author;
        this.eventType = builder.eventType;
        this.ipAddress = builder.ipAddress;
        this.eventMessage = builder.eventMessage;
        this.entries = builder.entries;
        this.entities = builder.entities;
        this.source = builder.source;
    }

    public Long getId() {
        return id;
    }

    @Override
    public Instant getTimestampInstant() {
        return timestamp;
    }

    @Override
    public AuditLogAuthorType getAuthorType() {
        return author.getType();
    }

    @Override
    public Long getAuthorId() {
        return author.getId();
    }

    @Override
    public String getAuthorName() {
        return author.getName();
    }

    @Override
    public AuditLogAuthor getAuthor() {
        return author;
    }

    @Override
    public AuditLogEventType getEventType() {
        return eventType;
    }

    @Override
    public String getIpAddress() {
        return ipAddress;
    }

    @Override
    public String getEventMessage() {
        return eventMessage;
    }

    @Override
    public AuditLogEventSource getSource() {
        return source;
    }

    @Override
    public Collection<ImmutableAuditLogEntry> getEntries() {
        return entries;
    }

    @Override
    public Collection<ImmutableAuditLogEntity> getEntities() {
        return entities;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        ImmutableAuditLogChangeset that = (ImmutableAuditLogChangeset) o;
        return Objects.equals(id, that.id) &&
                Objects.equals(timestamp, that.timestamp) &&
                Objects.equals(author, that.author) &&
                eventType == that.eventType &&
                Objects.equals(ipAddress, that.ipAddress) &&
                Objects.equals(eventMessage, that.eventMessage) &&
                Objects.equals(source, that.source) &&
                Objects.equals(entries, that.entries) &&
                Objects.equals(entities, that.entities);
    }

    @Override
    public int hashCode() {
        return Objects.hash(id, timestamp, author, eventType, ipAddress, eventMessage, source, entries, entities);
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("id", id)
                .add("timestamp", timestamp)
                .add("author", author)
                .add("eventType", eventType)
                .add("ipAddress", ipAddress)
                .add("eventMessage", eventMessage)
                .add("entries", entries)
                .add("entities", entities)
                .toString();
    }

    /**
     * @return an {@link ImmutableAuditLogChangeset} with the same properties as the given changeset.
     * Will avoid creating a copy if possible.
     */
    public static ImmutableAuditLogChangeset from(AuditLogChangeset changeset) {
        if (changeset instanceof ImmutableAuditLogChangeset) {
            return (ImmutableAuditLogChangeset) changeset;
        }

        return new ImmutableAuditLogChangeset.Builder(changeset).build();
    }

    private static boolean hasEntity(AuditLogEntityType entityType, Long entityId, String entityName) {
        return entityId != null || entityType != null || entityName != null;
    }

    /**
     * Builder for {@link ImmutableAuditLogChangeset}.
     *
     * Since 3.2.0 this class introduces special semantics for primary entities. The methods
     * <ul>
     * <li>{@link ImmutableAuditLogChangeset.Builder#setEntityId(Long)},</li>
     * <li>{@link ImmutableAuditLogChangeset.Builder#setEntityName(String)},</li>
     * <li>{@link ImmutableAuditLogChangeset.Builder#setEntityName(String)}</li>
     * </ul>
     * operate on an implicit primary entity. When building a concrete instance of {@link ImmutableAuditLogChangeset}
     * the entity will be prepended to the collection of entities and marked as primary.
     */
    public static class Builder {
        private Long id;
        private Instant timestamp;
        private AuditLogAuthor author = new ImmutableAuditLogAuthor(null, null, null);
        private AuditLogEventType eventType;
        private AuditLogEntityType entityType;
        private Long entityId;
        private String entityName;
        private String ipAddress;
        private String eventMessage;
        private AuditLogEventSource source = AuditLogEventSource.MANUAL;
        private Set<ImmutableAuditLogEntry> entries = new HashSet<>();
        private Set<ImmutableAuditLogEntity> entities = new HashSet<>();

        public Builder() {
        }

        public Builder(AuditLogChangeset changeset) {
            this.id = changeset.getId();
            this.timestamp = changeset.getTimestampInstant();
            this.author = changeset.getAuthor();
            this.eventType = changeset.getEventType();
            this.ipAddress = changeset.getIpAddress();
            this.eventMessage = changeset.getEventMessage();
            this.source = changeset.getSource();
            this.entries = changeset.getEntries().stream()
                    .map(ImmutableAuditLogEntry.Builder::new)
                    .map(ImmutableAuditLogEntry.Builder::build)
                    .collect(Collectors.toSet());
            this.entities = changeset.getEntities().stream().map(ImmutableAuditLogEntity::from).collect(Collectors.toSet());
        }

        public Builder setId(Long id) {
            this.id = id;
            return this;
        }

        public Builder setTimestamp(Instant timestamp) {
            this.timestamp = timestamp;
            return this;
        }
        public Builder setTimestamp(@Nonnull Date timestamp) {
            this.timestamp = timestamp.toInstant();
            return this;
        }

        public Builder setTimestamp(@Nonnull Long timestamp) {
            this.timestamp = Instant.ofEpochMilli(timestamp);
            return this;
        }

        @Deprecated
        public Builder setAuthorType(AuditLogAuthorType authorType) {
            this.author = new ImmutableAuditLogAuthor(author.getId(), author.getName(), authorType);
            return this;
        }

        @Deprecated
        public Builder setAuthorId(Long authorId) {
            this.author = new ImmutableAuditLogAuthor(authorId, author.getName(), author.getType());
            return this;
        }

        @Deprecated
        public Builder setAuthorName(String authorName) {
            this.author  = new ImmutableAuditLogAuthor(author.getId(), authorName, author.getType());
            return this;
        }

        public Builder setAuthor(AuditLogAuthor author) {
            this.author = new ImmutableAuditLogAuthor(author);
            return this;
        }

        public Builder setEventType(AuditLogEventType eventType) {
            this.eventType = eventType;
            return this;
        }

        /**
         * This method will set the type of the implicit primary entity. If this value is set to a non-null value,
         * an implicit primary entity will be prepended to the entities when building the changeset.
         * @param entityType the type of the implicit primary entity
         * @return the builder instance
         *
         * @deprecated use {@link ImmutableAuditLogChangeset.Builder#addEntity(ImmutableAuditLogEntity)} instead
         */
        @Deprecated
        public Builder setEntityType(AuditLogEntityType entityType) {
            this.entityType = entityType;
            return this;
        }

        /**
         * This method will set the identifier of the implicit primary entity. If this value is set to a non-null value,
         * an implicit primary entity will be prepended to the entities when building the changeset.
         * @param entityId the id of the implicit primary entity
         * @return the builder instance
         *
         * @deprecated use {@link ImmutableAuditLogChangeset.Builder#addEntity(ImmutableAuditLogEntity)} instead
         */
        @Deprecated
        public Builder setEntityId(Long entityId) {
            this.entityId = entityId;
            return this;
        }

        /**
         * This method will set the name of the implicit primary entity. If this value is set to a non-null value,
         * an implicit primary entity will be prepended to the entities when building the changeset.
         * @param entityName the name of the implicit primary entity
         * @return the builder instance
         *
         * @deprecated use {@link ImmutableAuditLogChangeset.Builder#addEntity(ImmutableAuditLogEntity)} instead
         */
        @Deprecated
        public Builder setEntityName(String entityName) {
            this.entityName = entityName;
            return this;
        }

        public Builder setIpAddress(String ipAddress) {
            this.ipAddress = ipAddress;
            return this;
        }

        public Builder setEventMessage(String eventMessage) {
            this.eventMessage = eventMessage;
            return this;
        }

        public Builder setSource(AuditLogEventSource source) {
            this.source = source;
            return this;
        }

        /**
         * deprecated Use {@link #setEntries(Collection)} instead. Since v3.2.0.
         */
        @Deprecated
        public Builder setEntries(List<ImmutableAuditLogEntry> entries) {
            this.entries = new HashSet<>(entries);
            return this;
        }

        /**
         * @since 3.2.0
         */
        public Builder setEntries(Collection<? extends AuditLogEntry> entries) {
            this.entries = entries.stream().map(entry -> new ImmutableAuditLogEntry.Builder(entry).build()).collect(Collectors.toSet());;
            return this;
        }

        /**
         * @since 3.2.0
         */
        public Builder setEntities(Collection<? extends AuditLogEntity> entities) {
            this.entities = entities.stream().map(ImmutableAuditLogEntity::from).collect(Collectors.toSet());
            return this;
        }

        public Builder addEntry(AuditLogEntry entry) {
            entries.add(ImmutableAuditLogEntry.from(entry));
            return this;
        }

        public Builder addEntry(ImmutableAuditLogEntry entry) {
            entries.add(entry);
            return this;
        }

        public Builder addEntries(Collection<AuditLogEntry> entry) {
            entries.addAll(entry.stream()
                    .map(e -> new ImmutableAuditLogEntry.Builder(e).build())
                    .collect(Collectors.toList())
            );
            return this;
        }

        public Builder addEntity(AuditLogEntity entity) {
            entities.add(ImmutableAuditLogEntity.from(entity));
            return this;
        }

        public Builder addEntity(ImmutableAuditLogEntity entity) {
            entities.add(entity);
            return this;
        }

        /**
         * @return an instance of {@link ImmutableAuditLogChangeset} from the values set in this builder. If entityId,
         * entityName or entityType are set, an instance of {@link ImmutableAuditLogEntity} will be prepended
         * to entities with the data from these fields and the primary value set to true.
         */
        public ImmutableAuditLogChangeset build() {
            if (hasEntity(entityType, entityId, entityName)) {
                entities.add(
                        new ImmutableAuditLogEntity.Builder()
                                .setEntityType(entityType)
                                .setEntityId(entityId)
                                .setEntityName(entityName)
                                .setPrimary()
                                .build()
                );
            }
            return new ImmutableAuditLogChangeset(this);
        }
    }

}
