package com.atlassian.confluence.plugins.createcontent.rest;

import com.atlassian.confluence.languages.LocaleManager;
import com.atlassian.confluence.pages.BlogPost;
import com.atlassian.confluence.pages.Page;
import com.atlassian.confluence.pages.PageManager;
import com.atlassian.confluence.plugins.createcontent.ContentBlueprintManager;
import com.atlassian.confluence.plugins.createcontent.api.exceptions.ResourceErrorType;
import com.atlassian.confluence.plugins.createcontent.exceptions.ResourceException;
import com.atlassian.confluence.plugins.createcontent.extensions.UserBlueprintConfigManager;
import com.atlassian.confluence.plugins.createcontent.impl.ContentBlueprint;
import com.atlassian.confluence.plugins.createcontent.rest.entities.CreateDialogWebItemEntity;
import com.atlassian.confluence.security.PermissionManager;
import com.atlassian.confluence.security.SpacePermissionManager;
import com.atlassian.confluence.spaces.Space;
import com.atlassian.confluence.spaces.SpaceManager;
import com.atlassian.confluence.user.ConfluenceUser;
import com.atlassian.confluence.util.i18n.DocumentationBean;
import com.atlassian.confluence.util.i18n.DocumentationBeanFactory;
import com.atlassian.confluence.util.i18n.I18NBean;
import com.atlassian.confluence.util.i18n.I18NBeanFactory;
import com.atlassian.plugin.ModuleCompleteKey;
import com.atlassian.plugins.rest.common.security.AnonymousAllowed;
import org.apache.commons.lang3.StringUtils;

import javax.ws.rs.DELETE;
import javax.ws.rs.FormParam;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Response;
import java.util.List;
import java.util.UUID;

@Path("/blueprints")
public class BlueprintResource extends AbstractRestResource {
    public static final String PARAM_SPACE_KEY = "spaceKey";
    public static final String PARAM_PAGE_TITLE = "pageTitle";
    public static final String PARAM_CONTENT_BLUEPRINT_ID = "contentBlueprintId";
    public static final String PARAM_SKIP = "skip";
    public static final String PARAM_ID = "id";
    public static final String PARAM_BP_MODULE_COMPLETE_KEY = "blueprintModuleCompleteKey";
    public static final String PARAM_KEY = "key";

    private final PermissionManager permissionManager;
    private final PageManager pageManager;
    private final LocaleManager localeManager;
    private final I18NBeanFactory i18NBeanFactory;
    private final DocumentationBeanFactory documentationBeanFactory;
    private final UserBlueprintConfigManager userBlueprintConfigManager;
    private final ContentBlueprintManager contentBlueprintManager;
    private final BlueprintWebItemService webItemService;

    public BlueprintResource(SpaceManager spaceManager, PermissionManager permissionManager,
                             PageManager pageManager, LocaleManager localeManager, I18NBeanFactory i18NBeanFactory,
                             DocumentationBeanFactory documentationBeanFactory, UserBlueprintConfigManager userBlueprintConfigManager,
                             ContentBlueprintManager contentBlueprintManager, final BlueprintWebItemService webItemService,
                             SpacePermissionManager spacePermissionManager) {
        super(permissionManager, spaceManager, spacePermissionManager);

        this.permissionManager = permissionManager;
        this.pageManager = pageManager;
        this.localeManager = localeManager;
        this.i18NBeanFactory = i18NBeanFactory;
        this.documentationBeanFactory = documentationBeanFactory;
        this.userBlueprintConfigManager = userBlueprintConfigManager;
        this.contentBlueprintManager = contentBlueprintManager;
        this.webItemService = webItemService;
    }

    @GET
    @Path("web-items")
    @AnonymousAllowed
    public List<CreateDialogWebItemEntity> getWebItems(@QueryParam(PARAM_SPACE_KEY) String spaceKey) {
        checkEmptyParameter(spaceKey, PARAM_SPACE_KEY);

        final Space space = getAndCheckSpace(spaceKey);
        final ConfluenceUser remoteUser = getUser();
        if (!userCanCreateInSpace(remoteUser, space)) {
            // This will only occur if the user hand-crafts a request, the user's permissions have changed since opening
            // the Create dialog, or if the User's session has expired and the remote user is null.
            throw new ResourceException("You are not permitted to see Create dialog items for space: " + spaceKey, Response.Status.FORBIDDEN,
                    remoteUser == null ? ResourceErrorType.PERMISSION_ANONYMOUS_CREATE : ResourceErrorType.PERMISSION_USER_CREATE, spaceKey);
        }

        final I18NBean i18NBean = i18NBeanFactory.getI18NBean(localeManager.getLocale(remoteUser));
        final DocumentationBean documentationBean = documentationBeanFactory.getDocumentationBean();

        return webItemService.getCreateContentWebItems(space, i18NBean, documentationBean, remoteUser);
    }

    private boolean userCanCreateInSpace(ConfluenceUser remoteUser, Space space) {
        return permissionManager.hasCreatePermission(remoteUser, space, Page.class)
                || permissionManager.hasCreatePermission(remoteUser, space, BlogPost.class);
    }

    /**
     * @return true if the user can create a page with the given title in the given space, false if not.
     */
    @GET
    @Path("can-create-page")
    @AnonymousAllowed
    public Boolean canCreatePage(@QueryParam(PARAM_SPACE_KEY) String spaceKey, @QueryParam(PARAM_PAGE_TITLE) String pageTitle) {
        checkEmptyParameter(spaceKey, PARAM_SPACE_KEY);
        checkEmptyParameter(pageTitle, PARAM_PAGE_TITLE);

        final Space space = getAndCheckSpace(spaceKey);

        return canCreatePage(space, pageTitle);
    }

    private boolean canCreatePage(Space space, String pageTitle) {
        if (!permissionManager.hasCreatePermission(getUser(), space, Page.class))
            return false;

        return pageManager.getPage(space.getKey(), pageTitle) == null;
    }

    /**
     * Called when the User indicates via checkbox that they don't want to see the How-to-Use Blueprint dialog page
     * again.
     */
    @POST
    @Path("skip-how-to-use")
    public void skipHowToUse(@FormParam(PARAM_CONTENT_BLUEPRINT_ID) UUID contentBlueprintId,
                             @FormParam(PARAM_SKIP) Boolean skip) {
        checkNullParameter(contentBlueprintId, PARAM_CONTENT_BLUEPRINT_ID);
        checkNullParameter(skip, PARAM_SKIP);

        userBlueprintConfigManager.setSkipHowToUse(getUser(), contentBlueprintId, skip);
    }

    // FIXME - shouldn't need the "get" part of the path.
    @GET
    @Path("get/{" + PARAM_ID + "}")
    public ContentBlueprint get(@PathParam(PARAM_ID) UUID id) {
        checkNullParameter(id, PARAM_ID);

        return contentBlueprintManager.getById(id);
    }

    @GET
    @Path("get")
    public ContentBlueprint get(@QueryParam(PARAM_BP_MODULE_COMPLETE_KEY) String blueprintModuleCompleteKey) {
        checkEmptyParameter(blueprintModuleCompleteKey, PARAM_BP_MODULE_COMPLETE_KEY);

        return contentBlueprintManager.getPluginBackedContentBlueprint(new ModuleCompleteKey(blueprintModuleCompleteKey), null);
    }

    @GET
    @Path("list")
    public List<ContentBlueprint> getAllContentBlueprints() {
        return contentBlueprintManager.getAll();
    }

    // TODO - should be a POST?
    @PUT
    @Path("create")
    public ContentBlueprint create(ContentBlueprint blueprint) {
        checkNullEntity(blueprint);

        String spaceKey = blueprint.getSpaceKey();
        if (StringUtils.isNotBlank(spaceKey))
            checkSpaceAdminPermission(spaceKey);
        else
            checkAdminPermission();

        return contentBlueprintManager.create(blueprint);
    }

    @GET
    @Path("byKey/{" + PARAM_KEY + "}")
    public ContentBlueprint getByModuleCompleteKey(@PathParam(PARAM_KEY) String moduleCompleteKey) {
        checkEmptyParameter(moduleCompleteKey, PARAM_KEY);

        ModuleCompleteKey key = new ModuleCompleteKey(moduleCompleteKey);
        return contentBlueprintManager.getPluginBackedContentBlueprint(key, null);
    }

    @DELETE
    @Path("deleteAll")
    public Integer deleteAll() {
        checkAdminPermission();

        return contentBlueprintManager.deleteAll();
    }
}