package com.atlassian.confluence.plugins.createcontent;

import com.atlassian.bandana.BandanaManager;
import com.atlassian.confluence.plugin.descriptor.web.DefaultWebInterfaceContext;
import com.atlassian.confluence.plugins.createcontent.activeobjects.PluginBackedBlueprintAo;
import com.atlassian.confluence.plugins.createcontent.impl.PluginBackedBlueprint;
import com.atlassian.confluence.plugins.createcontent.model.BlueprintState;
import com.atlassian.confluence.setup.bandana.ConfluenceBandanaContext;
import com.atlassian.confluence.setup.bandana.KeyedBandanaContext;
import com.atlassian.confluence.spaces.Space;
import com.atlassian.confluence.user.ConfluenceUser;
import com.atlassian.plugin.ModuleCompleteKey;
import com.atlassian.plugin.PluginAccessor;
import com.atlassian.plugin.web.WebInterfaceManager;
import com.atlassian.plugin.web.descriptors.WebItemModuleDescriptor;
import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.base.Strings;
import com.google.common.collect.Collections2;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;

import javax.annotation.Nullable;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.UUID;

import static com.atlassian.confluence.plugins.createcontent.api.contextproviders.BlueprintContextKeys.BLUEPRINT_MODULE_KEY;

/**
 * Stores the state of blueprint plugin modules in Bandana.
 */
public abstract class AbstractBandanaBlueprintStateController {
    private final BandanaManager bandanaManager;
    private final WebInterfaceManager webInterfaceManager;
    private final PluginAccessor pluginAccessor;

    private static final Function<String, UUID> getUuidFromString = new Function<String, UUID>() {
        @Override
        public UUID apply(@Nullable String blueprintId) {
            return UUID.fromString(blueprintId);
        }
    };

    private static final Predicate<UUID> isUuidNull = new Predicate<UUID>() {
        @Override
        public boolean apply(final UUID input) {
            return input != null;
        }
    };

    private static final Function<UUID, String> getStringFromUuid = new Function<UUID, String>() {
        @Override
        public String apply(@Nullable UUID blueprintId) {
            return blueprintId.toString();
        }
    };

    protected AbstractBandanaBlueprintStateController(BandanaManager bandanaManager,
                                                      WebInterfaceManager webInterfaceManager, PluginAccessor pluginAccessor) {
        this.bandanaManager = bandanaManager;
        this.webInterfaceManager = webInterfaceManager;
        this.pluginAccessor = pluginAccessor;
    }

    protected void enableBlueprint(UUID blueprintId, Space space, String bandanaKey) {
        if (blueprintId == null)
            throw new IllegalArgumentException("Blueprint UUID is required.");

        KeyedBandanaContext bandanaContext = space == null ? new ConfluenceBandanaContext() : new SpaceBandanaContext(space);

        Set<String> disabledBlueprintsIds = (Set<String>) bandanaManager.getValue(bandanaContext,
                bandanaKey);

        if (disabledBlueprintsIds != null)
            disabledBlueprintsIds.remove(blueprintId.toString());

        bandanaManager.setValue(bandanaContext, bandanaKey, disabledBlueprintsIds);
    }

    protected void disableBlueprint(UUID blueprintId, Space space, String bandanaKey) {
        if (blueprintId == null)
            throw new IllegalArgumentException("blueprint UUID is required.");

        if (blueprintId.equals(BlueprintConstants.UUID_BLANK_PAGE) || blueprintId.equals(BlueprintConstants.UUID_BLOG_POST)) {
            throw new IllegalArgumentException("You cannot disable this blueprint.");
        }

        KeyedBandanaContext bandanaContext = space == null ? new ConfluenceBandanaContext() : new SpaceBandanaContext(space);

        Set<String> disabledBlueprintsIds = (Set<String>) bandanaManager.getValue(bandanaContext,
                bandanaKey);

        if (disabledBlueprintsIds == null)
            disabledBlueprintsIds = Sets.newHashSet();

        disabledBlueprintsIds.add(blueprintId.toString());

        bandanaManager.setValue(bandanaContext, bandanaKey, disabledBlueprintsIds);
    }

    protected void disableBlueprints(Set<UUID> blueprintIds, Space space, String bandanaKey) {
        if (blueprintIds == null || blueprintIds.isEmpty())
            return;

        KeyedBandanaContext bandanaContext = space == null ? new ConfluenceBandanaContext() : new SpaceBandanaContext(space);

        Set<String> disabledBlueprintsIds = (Set<String>) bandanaManager.getValue(bandanaContext,
                bandanaKey);

        if (disabledBlueprintsIds == null)
            disabledBlueprintsIds = Sets.newHashSet();

        final Collection<String> disableBlueprintIds = Collections2.transform(blueprintIds, getStringFromUuid);

        disabledBlueprintsIds.addAll(disableBlueprintIds);
        bandanaManager.setValue(bandanaContext, bandanaKey, disabledBlueprintsIds);
    }

    protected Set<UUID> getDisabledBlueprintIds(Space space, String bandanaKey) {
        Set<UUID> disabledBlueprintIds = Sets.newHashSet();
        // Get all global disabled blueprints
        disabledBlueprintIds.addAll(getDisabledPluginIds(new ConfluenceBandanaContext(), bandanaKey));

        // Get all space-level disabled blueprints
        if (space != null)
            disabledBlueprintIds.addAll(getDisabledPluginIds(new SpaceBandanaContext(space), bandanaKey));

        return disabledBlueprintIds;
    }

    protected Set<UUID> getDisabledPluginIds(KeyedBandanaContext context, String bandanaKey) {
        Set<String> disabledBlueprints = (Set<String>) bandanaManager.getValue(context, bandanaKey);

        if (disabledBlueprints == null || disabledBlueprints.isEmpty())
            return Sets.newHashSet();

        final Collection<UUID> disableBlueprintIds =
                Collections2.filter(
                        Collections2.transform(disabledBlueprints, getUuidFromString),
                        isUuidNull);

        return Sets.newHashSet(disableBlueprintIds);
    }

    protected Set<String> getDisabledSpaceBlueprintModuleCompleteKeys(Space space, String bandanaKey, final AoBackedManager<? extends PluginBackedBlueprint, ? extends PluginBackedBlueprintAo> blueprintManager) {
        final Set<UUID> disabledModuleIds = getDisabledBlueprintIds(space, bandanaKey);

        final Collection<String> globalDisabledModuleCompleteKeys =
                Collections2.filter(
                        Collections2.transform(disabledModuleIds, new Function<UUID, String>() {
                            @Override
                            public String apply(UUID blueprintId) {
                                PluginBackedBlueprint blueprint = blueprintManager.getById(blueprintId);
                                if (blueprint != null)
                                    return blueprint.getModuleCompleteKey();

                                return null;
                            }
                        }),
                        new Predicate<String>() {
                            @Override
                            public boolean apply(final String input) {
                                return !Strings.isNullOrEmpty(input);
                            }
                        });

        return Sets.newHashSet(globalDisabledModuleCompleteKeys);
    }

    protected Map<UUID, BlueprintState> buildBlueprintStateMap(String section, ConfluenceUser user, Space space, String bandanaKey, Collection<? extends PluginBackedBlueprint> blueprints) {
        final HashMap<UUID, BlueprintState> blueprintStateMap = Maps.newHashMap();

        final Collection<UUID> globallyDisabledBlueprintIds = getDisabledBlueprintIds(null, bandanaKey);
        final Collection<UUID> spaceDisabledBlueprintIds = space == null
                ? Collections.<UUID>emptySet()
                : getDisabledBlueprintIds(space, bandanaKey);
        final Collection<String> webInterfaceManagerDisplayableModules =
                getWebInterfaceManagerDisplayableBlueprintModuleKeys(section, user, space);

        for (final PluginBackedBlueprint blueprint : blueprints) {
            final String moduleCompleteKey = blueprint.getModuleCompleteKey();
            final UUID blueprintId = blueprint.getId();
            final boolean isPluginBacked = StringUtils.isNotBlank(moduleCompleteKey);

            BlueprintState blueprintState = new BlueprintState.Builder()
                    .disabledInPluginSystem(isPluginBacked && !pluginAccessor.isPluginModuleEnabled(moduleCompleteKey))
                    .disabledGlobally(globallyDisabledBlueprintIds.contains(blueprintId))
                    .disabledInSpace(spaceDisabledBlueprintIds.contains(blueprintId))
                    .disabledByWebInterfaceManager(
                            isPluginBacked && !webInterfaceManagerDisplayableModules.contains(moduleCompleteKey)
                    )
                    .build();

            blueprintStateMap.put(blueprintId, blueprintState);
        }

        return blueprintStateMap;
    }

    private Collection<String> getWebInterfaceManagerDisplayableBlueprintModuleKeys(final String section,
                                                                                    final ConfluenceUser user, final Space space) {
        final DefaultWebInterfaceContext webInterfaceContext = new DefaultWebInterfaceContext();
        webInterfaceContext.setSpace(space);
        webInterfaceContext.setCurrentUser(user);
        final Map<String, Object> webInterfaceContextMap = webInterfaceContext.toMap();
        final ImmutableSet.Builder<String> displayableBlueprintKeys = ImmutableSet.builder();
        for (WebItemModuleDescriptor webItemModuleDescriptor : webInterfaceManager.getDisplayableItems(section, webInterfaceContextMap)) {
            String blueprintKey = webItemModuleDescriptor.getParams().get(BLUEPRINT_MODULE_KEY.key());
            if (StringUtils.isBlank(blueprintKey))
                continue;

            displayableBlueprintKeys.add(getCompleteKey(webItemModuleDescriptor.getPluginKey(), blueprintKey));
        }

        return displayableBlueprintKeys.build();
    }

    private static String getCompleteKey(String pluginKey, String blueprintKey) {
        return new ModuleCompleteKey(pluginKey, blueprintKey).getCompleteKey();
    }

}
