package com.atlassian.confluence.compat.struts2.servletactioncontext;

import aQute.bnd.annotation.component.Component;
import com.atlassian.confluence.api.service.exceptions.ServiceException;
import com.google.common.base.Supplier;
import com.google.common.base.Suppliers;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.servlet.ServletConfig;
import javax.servlet.ServletContext;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

/**
 * This class lets plugins to be compatible with all supported versions of Confluence which are using ServletActionContext class.
 * Since 8.0 Confluence uses newer ServletActionContext which is struts2 variant.
 */
@Component
public class ServletActionContextCompatManager implements ServletActionContextCompat {

    private static final Logger log = LoggerFactory.getLogger(ServletActionContextCompatManager.class);
    private final Supplier<ServletActionContextCompat> delegate;

    public ServletActionContextCompatManager() {
        this(ServletActionContextCompatManager.class.getClassLoader());
    }

    public ServletActionContextCompatManager(ClassLoader classLoader) {
        delegate = Suppliers.memoize(() -> initialiseServletActionContextCompat(classLoader));
    }

    private ServletActionContextCompat initialiseServletActionContextCompat(ClassLoader classLoader) {
        ServletActionContextCompat internalDelegate;
        try {
            Class.forName("org.apache.struts2.ServletActionContext", false, classLoader);
            internalDelegate = new ServletActionContextStruts2AndWWCompat("org.apache.struts2.ServletActionContext", classLoader);
        } catch (ClassNotFoundException e) {
            log.debug("Could not find struts2 ServletActionContext, falling back to webwork ServletActionContext", e);
            try {
                internalDelegate = new ServletActionContextStruts2AndWWCompat("com.opensymphony.webwork.ServletActionContext", classLoader);
            } catch (ReflectiveOperationException ex) {
                throw new ServiceException("ServletActionContext couldn't be initialized.", ex);
            }
        } catch (ReflectiveOperationException e) {
            throw new ServiceException("ServletActionContext couldn't be initialized.", e);
        }
        return internalDelegate;
    }

    /**
     * Sets the HTTP servlet request object
     */
    public void setRequest(HttpServletRequest request) {
        try {
            delegate.get().setRequest(request);
        } catch (NullPointerException ex) {
            log.error("Couldn't set the request");
        }
    }

    /**
     * Gets the HTTP servlet request object
     */
    public HttpServletRequest getRequest() {
        try {
            return delegate.get().getRequest();
        } catch (NullPointerException ex) {
            log.error("Couldn't get the request");
        }
        return null;
    }

    /**
     * Sets the HTTP servlet response object
     */
    public void setResponse(HttpServletResponse response) {
        try {
            delegate.get().setResponse(response);
        } catch (NullPointerException ex) {
            log.error("Couldn't set the response");
        }
    }

    /**
     * Gets the HTTP servlet response object
     */
    public HttpServletResponse getResponse() {
        try {
            return delegate.get().getResponse();
        } catch (NullPointerException ex) {
            log.error("Couldn't get the response");
        }
        return null;
    }

    /**
     * Gets the servlet config
     * We can only get servletcontext from servletconfig.getServletContext().
     * Added this method for backward compatibility.
     */
    public ServletConfig getServletConfig() {
        try {
            return delegate.get().getServletConfig();
        } catch (NullPointerException ex) {
            log.error("Couldn't get the ServletConfig");
        }
        return null;
    }

    /**
     * Gets the servlet context
     */
    public ServletContext getServletContext() {
        try {
            return delegate.get().getServletContext();
        } catch (NullPointerException ex) {
            log.error("Couldn't get the ServletConfig");
        }
        return null;
    }

    /**
     * Sets the servlet config
     */
    public void setServletConfig(ServletConfig config) {
        try {
            delegate.get().setServletConfig(config);
        } catch (NullPointerException ex) {
            log.error("Couldn't set the ServletConfig");
        }
    }
}
