package com.atlassian.cache.compat;

import com.atlassian.annotations.PublicApi;
import com.atlassian.cache.compat.memory.MemoryCacheFactory;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;

/**
 * A component that makes it possible to obtain a cache factory in multiple versions
 * of Atlassian applications.
 * <p>
 * It is a known problem that this compatibility cache factory <em>does not</em> correctly
 * support the cluster-safe caches that Confluence 5.4 provides using the atlassian-cache
 * version 0.1 API.  Any plugin that requires cluster safety in that version of Confluence
 * will need to find another way to accomplish that.
 * </p>
 *
 * @since v1.0
 */
@PublicApi
public class CompatibilityCacheFactory implements CacheFactory
{
    private static final Logger LOG = LoggerFactory.getLogger(CompatibilityCacheFactory.class);

    private static final String CACHE_SETTINGS_CLASS = "com.atlassian.cache.CacheSettings";
    private static final String DELEGATING_CACHE_MANAGER_CLASS = "com.atlassian.cache.compat.delegate.DelegatingCacheFactory";

    private final CacheFactory delegate;

    public CompatibilityCacheFactory()
    {
        if (isAtlassianCache2xAvailable())
        {

            this.delegate = getDelegatingCacheFactory();
        }
        else
        {
            this.delegate = new MemoryCacheFactory();
        }
    }



    @Override
    public <V> CachedReference<V> getCachedReference(@Nonnull String name, @Nonnull Supplier<V> supplier)
    {
        return delegate.getCachedReference(name, supplier);
    }

    @Override
    public <V> CachedReference<V> getCachedReference(@Nonnull String name, @Nonnull Supplier<V> supplier, @Nonnull CacheSettings required)
    {
        return delegate.getCachedReference(name, supplier, required);
    }

    @Override
    public <V> CachedReference<V> getCachedReference(@Nonnull Class<?> owningClass, @Nonnull String name, @Nonnull Supplier<V> supplier)
    {
        return delegate.getCachedReference(owningClass, name, supplier);
    }

    @Override
    public <V> CachedReference<V> getCachedReference(@Nonnull Class<?> owningClass, @Nonnull String name, @Nonnull Supplier<V> supplier, @Nonnull CacheSettings required)
    {
        return delegate.getCachedReference(owningClass, name, supplier, required);
    }

    @Override
    public <K, V> Cache<K, V> getCache(@Nonnull String name)
    {
        return delegate.getCache(name);
    }

    @Override
    public <K, V> Cache<K, V> getCache(@Nonnull Class<?> owningClass, @Nonnull String name)
    {
        return delegate.getCache(owningClass, name);
    }

    @Override
    public <K, V> Cache<K, V> getCache(@Nonnull String name, CacheLoader<K, V> loader)
    {
        return delegate.getCache(name, loader);
    }

    @Override
    public <K, V> Cache<K, V> getCache(@Nonnull String name, CacheLoader<K, V> loader, @Nonnull CacheSettings required)
    {
        return delegate.getCache(name, loader, required);
    }

    @Override
    @Deprecated
    public <K, V> Cache<K, V> getCache(@Nonnull String name, @Nonnull Class<K> keyType, @Nonnull Class<V> valueType)
    {
        return delegate.getCache(name, keyType, valueType);
    }



    CacheFactory getDelegate()
    {
        return delegate;
    }

    /**
     * Uses reflection to wrap the real CacheManager in our delegation layer.  Since we have already called
     * {@link #isAtlassianCache2xAvailable()} to check for the right level of the API before we do this,
     * it should always succeed.  Reflection keeps us from referencing DelegatingCacheFactory directly, which
     * means that the atlassian-cache-api classes are not accessed until that class is initialized, which in turn
     * is not done until after we have used ClassLoader to check the API level.
     */
    CacheFactory getDelegatingCacheFactory()
    {
        try
        {
            return (CacheFactory)getClass().getClassLoader().loadClass(DELEGATING_CACHE_MANAGER_CLASS).newInstance();
        }
        catch (Exception e)
        {
            throw new IllegalStateException("Unexpectedly failed to initialize delegation to atlassian-cache-api", e);
        }
        catch (LinkageError e)
        {
            throw new IllegalStateException("An unexpected binary incompatibility prevented delegation to atlassian-cache-api", e);
        }
    }


    /**
     * CacheSettings did not exist before atlassian-cache 2.x, so use it to make sure we see the binary compatible
     * atlassian-cache-api before we try to use it.
     */
    boolean isAtlassianCache2xAvailable()
    {
        try
        {
            getClass().getClassLoader().loadClass(CACHE_SETTINGS_CLASS);
            return true;
        }
        catch (ClassNotFoundException e)
        {
            LOG.debug("ClassLoader is not a defined class; using non-cluster-safe implementation", e);
        }
        return false;
    }
}
