package com.atlassian.cache.impl;

import com.atlassian.cache.CacheEntryListener;

import java.util.Set;
import java.util.concurrent.CopyOnWriteArraySet;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * A utility class which:
 * <ul>
 *     <li>Manages a set of {@code CacheEntryListener}s</li>
 *     <li>Provides a way to notify these listeners about different events</li>
 * </ul>
 */
public class DefaultCacheEntryListenerSupport<K, V> implements CacheEntryListenerSupport<K, V>
{
    private final Set<CacheEntryListener<K, V>> valueListeners = new CopyOnWriteArraySet<>();
    private final Set<CacheEntryListener<K, V>> valuelessListeners = new CopyOnWriteArraySet<>();
    private final CacheEntryNotificationSupport notificationSupport = CacheEntryNotificationSupport.getInstance();

    public void add(CacheEntryListener<K, V> listener, boolean includeValues)
    {
        (includeValues ? valueListeners : valuelessListeners)
                .add(checkNotNull(listener));
    }

    public void remove(CacheEntryListener<K, V> listener)
    {
        valueListeners.remove(checkNotNull(listener));
        valuelessListeners.remove(checkNotNull(listener));
    }

    public void notifyAdd(K key, V value)
    {
        notificationSupport.notifyAdd(valueListeners, new DefaultCacheEntryEvent<>(key, value, null));
        notificationSupport.notifyAdd(valuelessListeners, new DefaultCacheEntryEvent<>(key));
    }

    public void notifyEvict(K key, V oldValue)
    {
        notificationSupport.notifyEvict(valueListeners, new DefaultCacheEntryEvent<>(key, null, oldValue));
        notificationSupport.notifyEvict(valuelessListeners, new DefaultCacheEntryEvent<>(key));
    }

    public void notifyRemove(K key, V oldValue)
    {
        notificationSupport.notifyRemove(valueListeners, new DefaultCacheEntryEvent<>(key, null, oldValue));
        notificationSupport.notifyRemove(valuelessListeners, new DefaultCacheEntryEvent<>(key));
    }

    public void notifyUpdate(K key, V value, V oldValue)
    {
        notificationSupport.notifyUpdate(valueListeners, new DefaultCacheEntryEvent<>(key, value, oldValue));
        notificationSupport.notifyUpdate(valuelessListeners, new DefaultCacheEntryEvent<>(key));
    }
}
