package com.atlassian.cache;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

/**
 * Maintins a mapping of name,keyType,valueType -> Cache.  This class ensures typesafety for Caches returned via the
 * {@link #getCache(String, Class, Class)} method.
 *
 * @since v1.0
 */
public class DefaultCacheManager implements CacheManager
{
    //name -> CacheHolder map.  Entries should never be removed from this map, since it will break concurrent consistency
    private final ConcurrentMap<String, CacheHolder<?, ?>> caches = new ConcurrentHashMap<String, CacheHolder<?, ?>>();

    private final List<String> nonFlushableCacheNames;
    private final CacheProvider cacheProvider;

    public DefaultCacheManager(final CacheProvider cacheProvider)
    {
        this(cacheProvider, Collections.<String>emptyList());
    }

    public DefaultCacheManager(final CacheProvider cacheProvider, final List<String> nonFlushableCacheNames)
    {
        this.cacheProvider = cacheProvider;
        this.nonFlushableCacheNames = new ArrayList<String>(nonFlushableCacheNames);
    }

    public Collection<Cache<?, ?>> getCaches()
    {
        final Set<Cache<?, ?>> ret = new HashSet<Cache<?, ?>>();
        for (CacheHolder<?, ?> cacheHolder : caches.values())
        {
            ret.add(cacheHolder.getCache());
        }
        return ret;
    }

    public void flushCaches()
    {
        for (final CacheHolder cacheHolder : caches.values())
        {
            if (!nonFlushableCacheNames.contains(cacheHolder.getCache().getName()))
            {
                cacheHolder.getCache().removeAll();
            }
        }
    }

    public <K, V> Cache<K, V> getCache(final String name)
    {
        return getCache(name, null, null);
    }

    public <K, V> Cache<K, V> getCache(final String name, final Class<K> keyType, final Class<V> valueType)
    {
        if (!caches.containsKey(name))
        {
            caches.putIfAbsent(name, new CacheHolder<K, V>(name, keyType, valueType));
        }
        final CacheHolder<?, ?> holder = caches.get(name);

        if (holder.isTypeMatch(keyType, valueType))
        {
            @SuppressWarnings ("unchecked")
            final Cache<K, V> cache = (Cache<K, V>) holder.getCache();
            return cache;
        }
        //shouldn't get here, since the isTypeMatch method will throw a ClassCastException if the types are incorrect.
        throw new RuntimeException("Unreachable");
    }

    private class CacheHolder<K, V>
    {
        private Cache<K, V> cache;
        private final Class<K> keyType;
        private final Class<V> valueType;
        private final String name;

        public CacheHolder(String name, final Class<K> keyType, final Class<V> valueType)
        {
            this.name = name;
            this.keyType = keyType;
            this.valueType = valueType;
        }

        public synchronized Cache<K, V> getCache()
        {
            if (cache == null)
            {
                cache = cacheProvider.createCache(name, keyType, valueType);
            }
            return cache;
        }

        public boolean isTypeMatch(final Class<?> keyType, final Class<?> valueType) throws ClassCastException
        {
            if (this.keyType != null && !this.keyType.equals(keyType))
            {
                throw new ClassCastException("Key type doesn't match. Expected '" + this.keyType + "' but received '" + keyType + "'");
            }
            if (this.valueType != null && !this.valueType.equals(valueType))
            {
                throw new ClassCastException("Value type doesn't match. Expected '" + this.valueType + "' but received '" + valueType + "'");
            }
            return true;
        }
    }
}
