package com.atlassian.bitbucket.user;

import com.atlassian.bitbucket.permission.Permission;

import javax.annotation.Nonnull;
import java.util.Set;

/**
 * Plugins may wish to perform operations as a particular user.
 * <p>
 * This interface allows plugins to perform operations as a particular user.
 */
public interface SecurityService {

    /**
     * Creates a custom security context that is not authenticated that can be used to perform operations. Permissions
     * that have been granted to the current user will no longer apply. Note that any
     * {@link #withPermission(Permission, String) escalated permissions} that are associated with the current security
     * context will still apply.
     *
     * @param reason a description of the reason for creating the custom security context. This description is used for
     *               logging.
     * @return the custom security context
     */
    @Nonnull
    EscalatedSecurityContext anonymously(@Nonnull String reason);

    /**
     * Creates a custom security context with no difference to the current security context but allows adding further
     * permissions on the returned {@link EscalatedSecurityContext}.
     *
     * @param reason a description of the reason for creating the custom security context. This description is used for
     *               logging.
     * @return the custom security context
     */
    @Nonnull
    EscalatedSecurityContext escalate(@Nonnull String reason);

    /**
     * Creates a custom security context that is authenticated as {@code user} that can be used to perform operations.
     * Note that any {@link #withPermission(Permission, String) escalated permissions} that are associated with the
     * current security context will still apply.
     *
     *
     * @param user the user to temporarily execute as
     * @param reason a description of the reason for creating the custom security context. This description is used for
     *               logging.
     * @return the custom security context
     */
    @Nonnull
    EscalatedSecurityContext impersonating(@Nonnull ApplicationUser user, @Nonnull String reason);

    /**
     * Creates a custom security context with elevated permissions that can be used to perform operations. The custom
     * context is still authenticated as the current user (if there is one).
     * Note that any escalated permissions associated with the current security context will still apply.
     *
     *
     * @param permission the permission to temporarily grant. This can be either a global or a resource permission. If
     *                   a resource permission is provided, the permission is granted to all resources of that type.
     * @param reason a description of the reason for creating the custom security context. This description is used for
     *               logging.
     * @return the custom security context
     */
    @Nonnull
    EscalatedSecurityContext withPermission(@Nonnull Permission permission, @Nonnull String reason);

    /**
     * Creates a custom security context with elevated permissions that can be used to perform operations. The custom
     * context is still authenticated as the current user (if there is one).
     * Note that any escalated permissions that are associated with the current security context will still apply.
     *
     * @param permission the permission to temporarily grant. This has to be a resource permission that is valid for the
     *                   provided resource (e.g. REPO_READ for a Repository).
     * @param resource the resource to temporarily grant the permission on
     * @param reason a description of the reason for creating the custom security context. This description is used for
     *               logging.
     * @return the custom security context
     */
    @Nonnull
    EscalatedSecurityContext withPermission(@Nonnull Permission permission, @Nonnull Object resource, @Nonnull String reason);

    /**
     * Creates a custom security context with elevated permissions that can be used to perform operations. The custom
     * context is still authenticated as the current user (if there is one).
     * Note that any {@link #withPermission(Permission, String) escalated permissions} that are associated with the
     * current security context will still apply.
     *
     *
     * @param permissions the permissions to temporarily grant. This can be a mix of global and resource permissions. If
     *                    a resource permission is provided, the permission is granted to all resources of that type.
     * @param reason a description of the reason for creating the custom security context. This description is used for
     *               logging.
     * @return the custom security context
     */
    @Nonnull
    EscalatedSecurityContext withPermissions(@Nonnull Set<Permission> permissions, @Nonnull String reason);
}
