package com.atlassian.bitbucket.scm.bulk;

import com.atlassian.bitbucket.scm.AbstractCommandParameters;

import javax.annotation.Nonnull;
import java.util.Optional;

import static java.util.Objects.requireNonNull;
import static java.util.Optional.ofNullable;

/**
 * Parameters for {@link ScmBulkContentCommandFactory#contents} call.
 *
 * @since 4.2
 */
public class BulkContentCommandParameters extends AbstractCommandParameters {

    private final String sinceCommitId;
    private final String untilCommitId;

    private BulkContentCommandParameters(Builder builder) {
        sinceCommitId = builder.sinceCommitId;
        untilCommitId = builder.untilCommitId;
    }

    /**
     * @return optional ID of the "since" commit; if specified only contents that changed since that commit ID are
     *         streamed, otherwise all contents are streamed
     */
    @Nonnull
    public Optional<String> getSinceCommitId() {
        return ofNullable(sinceCommitId);
    }

    /**
     * @return ID of the commit whose file contents should be streamed
     */
    @Nonnull
    public String getUntilCommitId() {
        return untilCommitId;
    }

    public static class Builder {

        private final String untilCommitId;
        private String sinceCommitId;

        public Builder(@Nonnull String untilCommitId) {
            this.untilCommitId = requireNonNull(untilCommitId, "untilCommitId");
        }

        @Nonnull
        public Builder sinceCommitId(@Nonnull String value) {
            sinceCommitId = requireNonNull(value, "sinceCommitId");
            return this;
        }

        @Nonnull
        public BulkContentCommandParameters build() {
            return new BulkContentCommandParameters(this);
        }
    }
}
