package com.atlassian.bitbucket.scm;

import com.atlassian.bitbucket.io.InputSupplier;
import com.atlassian.bitbucket.repository.Branch;
import com.atlassian.bitbucket.user.ApplicationUser;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.io.InputStream;
import java.util.Optional;

import static com.google.common.base.MoreObjects.firstNonNull;
import static java.util.Objects.requireNonNull;
import static java.util.Optional.ofNullable;

/**
 * Parameters for {@link ScmExtendedCommandFactory#editFile(EditFileCommandParameters) editing} a file.
 *
 * @since 4.13
 */
public class EditFileCommandParameters extends AbstractCommandParameters {

    private final ApplicationUser author;
    private final Branch branch;
    private final ApplicationUser committer;
    private final InputSupplier<InputStream> content;
    private final String message;
    private final String path;
    private final String sourceCommitId;
    private final String targetBranch;

    public EditFileCommandParameters(Builder builder) {
        author = builder.author;
        branch = builder.branch;
        committer = firstNonNull(builder.committer, builder.author);
        content = builder.content;
        message = builder.message;
        path = builder.path;
        sourceCommitId = builder.sourceCommitId;
        targetBranch = requireNonNull(builder.targetBranch, "targetBranch");
    }

    @Nonnull
    public ApplicationUser getAuthor() {
        return author;
    }

    /**
     * @return the branch from which the file edit should be performed
     */
    @Nonnull
    public Optional<Branch> getBranch() {
        return ofNullable(branch);
    }

    /**
     * @since 7.13
     */
    @Nonnull
    public ApplicationUser getCommitter() {
        return committer;
    }

    @Nonnull
    public InputSupplier<InputStream> getContent() {
        return content;
    }

    @Nonnull
    public String getMessage() {
        return message;
    }

    @Nonnull
    public String getPath() {
        return path;
    }

    @Nonnull
    public Optional<String> getSourceCommitId() {
        return ofNullable(sourceCommitId);
    }

    /**
     * @return the name of the target branch
     *
     * @since 5.2
     */
    @Nonnull
    public String getTargetBranch() {
        return targetBranch;
    }

    public static class Builder {

        private final Branch branch;
        private final String path;

        private ApplicationUser author;
        private ApplicationUser committer;
        private InputSupplier<InputStream> content;
        private String message;
        private String sourceCommitId;
        private String targetBranch;

        public Builder(@Nullable Branch branch, @Nonnull String path) {
            this.branch = branch;
            this.path = requireNonBlank(path, "path");

            if (branch != null) {
                targetBranch = branch.getId();
            }
        }

        @Nonnull
        public Builder author(@Nonnull ApplicationUser value) {
            author = requireNonNull(value, "author");
            return this;
        }

        @Nonnull
        public EditFileCommandParameters build() {
            requireNonNull(author, "author");
            requireNonNull(content, "content");
            requireNonBlank(message, "message");
            return new EditFileCommandParameters(this);
        }

        /**
         * @since 7.13
         */
        @Nonnull
        public Builder committer(@Nonnull ApplicationUser value) {
            committer = requireNonNull(value, "committer");
            return this;
        }

        @Nonnull
        public Builder content(@Nonnull InputSupplier<InputStream> value) {
            content = requireNonNull(value, "content");
            return this;
        }

        @Nonnull
        public Builder message(@Nonnull String value) {
            message = requireNonBlank(value, "message");
            return this;
        }

        @Nonnull
        public Builder sourceCommitId(@Nullable String value) {
            sourceCommitId = value;
            return this;
        }

        /**
         * @param value new value
         * @return {@code this}
         * @since 5.2
         */
        @Nonnull
        public Builder targetBranch(@Nonnull String value) {
            targetBranch = requireNonNull(value, "targetBranch");
            return this;
        }
    }
}
