package com.atlassian.bitbucket.scm;

import com.atlassian.bitbucket.ServiceException;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.time.Duration;
import java.util.concurrent.Callable;

import static java.util.Objects.requireNonNull;

/**
 * Wraps an operation which may be run {@link #call() synchronously}. Prior to performing the operation, timeouts may
 * be set to control how long the operation is allowed to run without producing output or processing input (the
 * its {@link #setIdleTimeout(long) idle timeout} or to provide an absolute limit how long the operation is allowed to
 * run (the {@link #setExecutionTimeout(long) execution timeout}.
 * <p>
 * When the operation completes, it will return an object of type {@code T}. If {@code T} is {@code Void}, the operation
 * will return {@code null} on completion. Typically this means the operation uses some form of callback, allowing the
 * caller to process the results as they are produced rather than returning them on completion.
 *
 * @param <T> the type of value that the operation will produce.
 */
public interface Command<T> extends Callable<T> {

    /**
     * Transforms this {@code Command} into an {@link AsyncCommand} which may be called asynchronously. Any timeouts
     * applied this command <i>will not be applied</i> to the resulting {@link AsyncCommand}.
     * <p>
     * Once an {@code Command} has been transformed for asynchronous use, it may no longer be used
     * {@link #call() synchronously}.
     *
     * @return an {@link AsyncCommand} for executing this {@code Command} asynchronously
     * @throws UnsupportedOperationException if the implementation cannot internally convert the operation for
     *                                       asynchronous execution
     */
    @Nonnull
    AsyncCommand<T> asynchronous();

    /**
     * Executes a command
     *
     * @return the result of the operation
     * @throws CommandCanceledException if command execution is canceled
     * @throws ServiceException if an error occurs while performing the operation
     */
    @Nullable
    @Override
    T call();

    /**
     * Sets the maximum time, in seconds, the operation is allowed to run. The operation will be aborted automatically
     * if it hasn't completed in this time <i>even if it is still producing output or processing input</i>.
     *
     * @param timeoutInSecs the maximum time the operation is allowed to run, in seconds
     * @throws UnsupportedOperationException if the implementation does not support configuring an execution timeout
     */
    void setExecutionTimeout(long timeoutInSecs);

    /**
     * Sets the maximum duration the operation is allowed to run. The operation will be aborted automatically
     * if it hasn't completed in this time <i>even if it is still producing output or processing input</i>.
     *
     * @param timeout the maximum duration the operation is allowed to run
     * @throws UnsupportedOperationException if the implementation does not support configuring an execution timeout
     * @since 5.1
     */
    default void setExecutionTimeout(@Nonnull Duration timeout) {
        setExecutionTimeout(requireNonNull(timeout, "timeout").getSeconds());
    }

    /**
     * Sets the maximum time, in seconds, the operation is allowed to run without <i>either</i> producing output or
     * processing input. Output may be produced on either the standard output or standard error streams.
     *
     * @param timeoutInSecs the maximum time the operation is allowed to be idle, in seconds
     * @throws UnsupportedOperationException if the implementation does not support configuring an idle timeout
     */
    void setIdleTimeout(long timeoutInSecs);

    /**
     * Sets the maximum duration the operation is allowed to run without <i>either</i> producing output or
     * processing input. Output may be produced on either the standard output or standard error streams.
     *
     * @param timeout the maximum duration the operation is allowed to be idle
     * @throws UnsupportedOperationException if the implementation does not support configuring an idle timeout
     * @since 5.1
     */
    default void setIdleTimeout(@Nonnull Duration timeout) {
        setIdleTimeout(requireNonNull(timeout, "timeout").getSeconds());
    }

    /**
     * Sets the {@link #setIdleTimeout(long) idle} and {@link #setExecutionTimeout(long) execution} timeouts, in
     * seconds, for the operation. This is useful for commands that do their processing up-front and may not produce
     * any output until processing is complete. Setting the idle timeout to the same value as the execution timeout
     * effectively disables the idle timeout.
     *
     * @param timeoutInSecs the maximum time the operation is allowed to run or be idle, in seconds
     * @since 5.1
     */
    default void setTimeout(long timeoutInSecs) {
        setExecutionTimeout(timeoutInSecs);
        setIdleTimeout(timeoutInSecs);
    }

    /**
     * Sets the {@link #setIdleTimeout(Duration) idle} and {@link #setExecutionTimeout(Duration) execution} timeouts
     * to the specified duration. This is useful for commands that do their processing up-front and may not produce
     * any output until processing is complete. Setting the idle timeout to the same value as the execution timeout
     * effectively disables the idle timeout.
     *
     * @param timeout the maximum duration the operation is allowed to run or be idle
     * @since 5.1
     */
    default void setTimeout(@Nonnull Duration timeout) {
        setExecutionTimeout(timeout);
        setIdleTimeout(timeout);
    }
}
