package com.atlassian.bitbucket.repository;

import com.atlassian.bitbucket.util.BuilderSupport;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import static java.util.Objects.requireNonNull;
import static com.google.common.base.Preconditions.checkState;

/**
 * A request for metadata associated with a collection of {@link Ref}.
 */
public class RefMetadataRequest {

    private final Map<String, Object> context;
    private final Ref baseRef;
    private final Set<Ref> refs;
    private final Repository repository;

    private RefMetadataRequest(Builder builder) {
        this.context = builder.context.build();
        this.baseRef = builder.baseRef;
        this.refs = builder.refs.build();
        this.repository = builder.repository;

        checkState(!refs.isEmpty(), "At least one ref must be specified");
        checkState(repository != null, "A repository is required");
    }

    /**
     * @return the base ref the refs should be compared to (for the providers that uses that information)
     */
    @Nullable
    public Ref getBaseRef() {
        return baseRef;
    }

    /**
     * @return the collection of refs to find associated metadata
     */
    @Nonnull
    public Set<Ref> getRefs() {
        return refs;
    }

    /**
     * @return the repository the refs exist within
     */
    @Nonnull
    public Repository getRepository() {
        return repository;
    }

    /**
     * @return a map of attributes associated with the context.
     */
    @Nonnull
    public Map<String, Object> getContext() {
        return context;
    }

    /**
     * A builder for {@link RefMetadataRequest}
     */
    public static class Builder extends BuilderSupport {

        private final ImmutableMap.Builder<String, Object> context;
        private final ImmutableSet.Builder<Ref> refs;
        private Ref baseRef;
        private Repository repository;

        public Builder() {
            context = ImmutableMap.builder();
            refs = ImmutableSet.builder();
        }

        public Builder(RefMetadataRequest request) {
            this();
            baseRef(request.getBaseRef()).refs(request.getRefs()).repository(request.getRepository());
        }

        public RefMetadataRequest build() {
            return new RefMetadataRequest(this);
        }

        public Builder baseRef(Ref baseRef) {
            this.baseRef = baseRef;

            return this;
        }

        public Builder refs(Iterable<? extends Ref> values) {
            addIf(Objects::nonNull, refs, values);

            return this;
        }

        public Builder refs(Ref value, Ref... values) {
            addIf(Objects::nonNull, refs, value, values);

            return this;
        }

        public Builder repository(Repository value) {
            repository = requireNonNull(value, "repository");

            return this;
        }

        @Nonnull
        public Builder context(@Nonnull Map<String, Object> value) {
            context.putAll(requireNonNull(value, "context"));

            return this;
        }
    }

}
