package com.atlassian.bitbucket.pull.reviewer;

import com.atlassian.bitbucket.user.ApplicationUser;
import com.atlassian.bitbucket.util.BuilderSupport;
import com.google.common.collect.ImmutableSet;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Arrays;
import java.util.Objects;
import java.util.Set;

/**
 * @since 7.13
 * Request object used for {@link ReviewerGroupService#update(ReviewerGroupUpdateRequest) updating a reviewer group}.
 */
public class ReviewerGroupUpdateRequest {

    private final String description;
    private final long id;
    private final String name;
    private final Set<ApplicationUser> users;

    private ReviewerGroupUpdateRequest(Builder builder) {
        description = builder.description;
        id = builder.id;
        name = builder.name;
        users = builder.users.build();
    }

    /**
     * @return the new description of a reviewer group. Passing an empty string ("") will set the reviewer group
     *         description to be null. If the description in this object is null then the reviewer group description
     *         will not be updated.
     */
    @Nullable
    public String getDescription() {
        return description;
    }

    /**
     * @return the id of the reviewer group
     */
    public long getId() {
        return id;
    }

    /**
     * @return the new name of a reviewer group
     */
    @Nonnull
    public String getName() {
        return name;
    }

    /**
     * @return a new set of reviewers that would replace the existing set. Since reviewer groups must contain at least
     *         one reviewer, an empty set would indicate that no changes are made to the existing set of reviewers.
     */
    @Nonnull
    public Set<ApplicationUser> getUsers() {
        return users;
    }

    public static class Builder extends BuilderSupport {

        private final long id;
        private final ImmutableSet.Builder<ApplicationUser> users;

        private String description;
        private String name;

        public Builder(ReviewerGroup reviewerGroup) {
            id = reviewerGroup.getId();
            description = reviewerGroup.getDescription();
            name = reviewerGroup.getName();
            users = ImmutableSet.builder();
        }

        @Nonnull
        public ReviewerGroupUpdateRequest build() {
            return new ReviewerGroupUpdateRequest(this);
        }

        @Nonnull
        public Builder description(@Nullable String value) {
            description = value;

            return this;
        }

        @Nonnull
        public Builder name(@Nonnull String value) {
            name = requireNonBlank(value, "name");

            return this;
        }

        @Nonnull
        public Builder users(@Nonnull ApplicationUser... values) {
            users.addAll(Arrays.asList(values));

            return this;
        }

        @Nonnull
        public Builder users(@Nonnull Iterable<ApplicationUser> values) {
            addIf(Objects::nonNull, users, values);

            return this;
        }
    }
}
