package com.atlassian.bitbucket.project;

import com.atlassian.bitbucket.permission.Permission;
import org.apache.commons.lang3.StringUtils;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Objects.requireNonNull;

/**
 * Request for searching projects. The purpose and behaviour of each field is described on its accessor.
 */
public class ProjectSearchRequest {

    private final String name;
    private final Permission permission;

    private ProjectSearchRequest(String name, Permission permission) {
        this.name = StringUtils.trimToNull(name);
        this.permission = permission;
    }

    /**
     * When set, limits returned {@link Project projects} to only those whose {@link Project#getName() names}
     * <i>contain</i> the provided value. Matching is performed in a case-insensitive manner, and will match
     * <i>anywhere</i> within projects' names.
     * <p>
     * Note: Values containing only whitespace are <i>ignored</i>, and will not be applied. Additionally, leading
     * and trailing whitespace are trimmed.
     *
     * @return filter text match against project names, or {@code null} to return projects regardless of name
     */
    @Nullable
    public String getName() {
        return name;
    }

    /**
     * When set, limits returned {@link Project projects} to only those for which the current user has the required
     * {@link Permission}. The provided value may be {@code null}, in which case projects will automatically be
     * limited to those for which the current user has {@link Permission#PROJECT_VIEW VIEW} permission.
     *
     * @return the required permission, or {@code null} to default to {@link Permission#PROJECT_VIEW VIEW}
     */
    @Nullable
    public Permission getPermission() {
        return permission;
    }

    /**
     * Retrieves a flag indicating whether {@link #getName() name} text has been set.
     *
     * @return {@code true} if {@link #getName()} is not {@code null} or whitespace; otherwise, {@code false}
     */
    public boolean hasName() {
        return StringUtils.isNotBlank(getName());
    }

    /**
     * Retrieves a flag indicating whether a specific {@link #getPermission() permission} has been set. If no explicit
     * permission has been set, the search will default to {@link Permission#PROJECT_VIEW VIEW}.
     *
     * @return {@code true} if {@link #getPermission()} is not {@code null}; otherwise, {@code false}
     */
    public boolean hasPermission() {
        return permission != null;
    }

    public static class Builder {

        private String name;
        private Permission permission;

        public Builder() {
        }

        /**
         * Constructs a new {@code Builder} which will copy initial values from the provided
         * {@link ProjectSearchRequest request}.
         *
         * @param request request to copy
         */
        public Builder(@Nonnull ProjectSearchRequest request) {
            this.name = requireNonNull(request, "request").getName();
            this.permission = request.getPermission();
        }

        /**
         * Assembles a new {@link ProjectSearchRequest} from the provided values.
         *
         * @return a new request
         */
        @Nonnull
        public ProjectSearchRequest build() {
            return new ProjectSearchRequest(name, permission);
        }

        /**
         * When set, limits returned {@link Project projects} to only those whose {@link Project#getName() names}
         * <i>contain</i> the provided value. Matching is performed in a case-insensitive manner, and will match
         * <i>anywhere</i> within projects' names.
         * <p>
         * Note: Values containing only whitespace are <i>ignored</i>, and will not be applied. Additionally, leading
         * and trailing whitespace are trimmed.
         *
         * @param value filter text match against project names, or {@code null} to return projects regardless of name
         * @return {@code this}
         */
        @Nonnull
        public Builder name(@Nullable String value) {
            name = value;

            return this;
        }

        /**
         * When set, limits returned {@link Project projects} to only those for which the current user has the required
         * {@link Permission}. The provided value may be {@code null}, in which case projects will automatically be
         * limited to those for which the current user has {@link Permission#PROJECT_VIEW VIEW} permission.
         *
         * @param value the required permission, or {@code null} to default to {@link Permission#PROJECT_VIEW VIEW}
         * @return {@code this}
         * @throws IllegalArgumentException if the provided {@link Permission} is not
         *                                  {@link Permission#isResource(Class) project related}
         */
        @Nonnull
        public Builder permission(@Nullable Permission value) {
            checkArgument(value == null || value.isResource(Project.class),
                    "The provided permission is not valid for projects");
            permission = value;

            return this;
        }
    }

}
