package com.atlassian.bitbucket.mesh;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import static java.util.Objects.requireNonNull;

/**
 * Describes a request to {@link MeshService#update update} properties for an existing {@link MeshNode}.
 *
 * @since 8.0
 */
public class UpdateMeshNodeRequest {

    private final long id;
    private final String name;
    private final String rpcUrl;
    private final MeshNode.State state;

    private UpdateMeshNodeRequest(Builder builder) {
        id = builder.id;
        name = builder.name;
        rpcUrl = builder.rpcUrl;
        state = builder.state;
    }

    /**
     * @return the ID of the node to update
     */
    public long getId() {
        return id;
    }

    /**
     * @return the node's new name, or {@code null} to retain its existing name unchanged
     */
    @Nullable
    public String getName() {
        return name;
    }

    /**
     * @return the node's new URL, or {@code null} to retain its existing URL unchanged
     */
    @Nullable
    public String getRpcUrl() {
        return rpcUrl;
    }

    /**
     * @return the node's new {@link MeshNode.State state}, or {@code null} to retain its existing state unchanged
     * @since 8.4
     */
    @Nullable
    public MeshNode.State getState() {
        return state;
    }

    public static class Builder {

        private final long id;

        private String name;
        private String rpcUrl;
        private MeshNode.State state;

        public Builder(@Nonnull MeshNode node) {
            this(requireNonNull(node, "node").getId());
        }

        public Builder(long id) {
            this.id = id;
        }

        @Nonnull
        public UpdateMeshNodeRequest build() {
            return new UpdateMeshNodeRequest(this);
        }

        @Nonnull
        public Builder name(@Nonnull String value) {
            name = requireNonNull(value, "name");
            return this;
        }

        @Nonnull
        public Builder rpcUrl(@Nonnull String value) {
            rpcUrl = requireNonNull(value, "rpcUrl");
            return this;
        }

        /**
         * @param value the node's new {@link MeshNode.State state}
         * @return the builder
         * @since 8.4
         */
        @Nonnull
        public Builder state(@Nonnull MeshNode.State value) {
            state = requireNonNull(value, "state");
            return this;
        }
    }
}
