package com.atlassian.bitbucket.hook.script;

import com.atlassian.bitbucket.hook.repository.RepositoryHookTrigger;
import com.atlassian.bitbucket.scope.Scope;
import com.atlassian.bitbucket.util.BuilderSupport;
import com.google.common.collect.ImmutableSet;

import javax.annotation.Nonnull;
import java.util.Arrays;
import java.util.Set;

import static java.util.Objects.requireNonNull;

/**
 * @since 6.2
 */
public class HookScriptSetConfigurationRequest {

    private final Scope scope;
    private final HookScript script;
    private final Set<String> triggerIds;

    private HookScriptSetConfigurationRequest(Builder builder) {
        scope = builder.scope;
        script = builder.script;
        triggerIds = builder.triggerIds.build();
    }

    @Nonnull
    public Scope getScope() {
        return scope;
    }

    @Nonnull
    public HookScript getScript() {
        return script;
    }

    @Nonnull
    public Set<String> getTriggerIds() {
        return triggerIds;
    }

    public static class Builder extends BuilderSupport {

        private final Scope scope;
        private final HookScript script;
        private final ImmutableSet.Builder<String> triggerIds;

        public Builder(@Nonnull HookScript script, @Nonnull Scope scope) {
            this.script = requireNonNull(script, "script");
            this.scope = requireNonNull(scope, "scope");

            triggerIds = ImmutableSet.builder();
        }

        @Nonnull
        public HookScriptSetConfigurationRequest build() {
            return new HookScriptSetConfigurationRequest(this);
        }

        @Nonnull
        public Builder triggerId(@Nonnull String value) {
            triggerIds.add(requireNonBlank(value, "triggerId"));

            return this;
        }

        @Nonnull
        public Builder triggerIds(@Nonnull Iterable<String> values) {
            requireNonNull(values, "triggerIds").forEach(this::triggerId);

            return this;
        }

        @Nonnull
        public Builder triggerIds(@Nonnull String value, @Nonnull String... values) {
            return triggerId(value)
                    .triggerIds(Arrays.asList(requireNonNull(values, "triggerIds")));
        }

        @Nonnull
        public Builder trigger(@Nonnull RepositoryHookTrigger trigger) {
            return triggerId(requireNonNull(trigger, "trigger").getId());
        }

        @Nonnull
        public Builder triggers(@Nonnull Iterable<RepositoryHookTrigger> values) {
            requireNonNull(values, "triggers").forEach(this::trigger);

            return this;
        }

        @Nonnull
        public Builder triggers(@Nonnull RepositoryHookTrigger value, @Nonnull RepositoryHookTrigger... values) {
            return trigger(value)
                    .triggers(Arrays.asList(requireNonNull(values, "triggers")));
        }
    }
}
