package com.atlassian.bitbucket.hook.repository;

import com.atlassian.bitbucket.pull.PullRequest;
import com.atlassian.bitbucket.pull.PullRequestRef;
import com.atlassian.bitbucket.repository.RefChange;
import com.atlassian.bitbucket.repository.RefChangeType;
import com.atlassian.bitbucket.repository.SimpleRefChange;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Optional;

import static java.util.Objects.requireNonNull;
import static java.util.Optional.ofNullable;

/**
 * Basic implementation of {@link PullRequestMergeHookRequest}
 *
 * @since 5.0
 */
public class SimplePullRequestMergeHookRequest extends AbstractRepositoryHookRequest
        implements PullRequestMergeHookRequest {

    private final String mergeHash;
    private final String message;
    private final PullRequest pullRequest;
    private final List<RefChange> refChanges;
    private final String strategyId;

    private SimplePullRequestMergeHookRequest(Builder builder) {
        super(builder);

        mergeHash = builder.mergeHash;
        message = builder.message;
        pullRequest = requireNonNull(builder.pullRequest, "pullRequest");
        strategyId = builder.strategyId;
        if (mergeHash == null) {
            refChanges = Collections.emptyList();
        } else {
            refChanges = Collections.singletonList(new SimpleRefChange.Builder()
                .from(pullRequest.getToRef())
                .toHash(mergeHash)
                .type(RefChangeType.UPDATE)
                .build());
        }
    }

    @Nonnull
    @Override
    public PullRequestRef getFromRef() {
        return pullRequest.getFromRef();
    }

    @Nonnull
    @Override
    public Optional<String> getMergeHash() {
        return ofNullable(mergeHash);
    }

    @Nonnull
    @Override
    public Optional<String> getMessage() {
        return ofNullable(message);
    }

    @Nonnull
    @Override
    public PullRequest getPullRequest() {
        return pullRequest;
    }

    /**
     * @since 5.5
     */
    @Nonnull
    @Override
    public Optional<String> getStrategyId() {
        return ofNullable(strategyId);
    }

    @Nonnull
    @Override
    public PullRequestRef getToRef() {
        return pullRequest.getToRef();
    }

    @Nonnull
    @Override
    public Collection<RefChange> getRefChanges() {
        return refChanges;
    }

    @Override
    public boolean isCrossRepository() {
        return pullRequest.isCrossRepository();
    }

    public static class Builder extends AbstractBuilder<Builder> {

        private final PullRequest pullRequest;

        private String mergeHash;
        private String message;
        private String strategyId;

        public Builder(@Nonnull PullRequest pullRequest) {
            super(requireNonNull(pullRequest, "pullRequest").getToRef().getRepository(),
                    StandardRepositoryHookTrigger.PULL_REQUEST_MERGE);

            this.pullRequest = pullRequest;
        }

        @Nonnull
        public SimplePullRequestMergeHookRequest build() {
            return new SimplePullRequestMergeHookRequest(this);
        }

        @Nonnull
        public Builder mergeHash(@Nullable String value) {
            mergeHash = value;
            return self();
        }

        @Nonnull
        public Builder message(@Nullable String value) {
            message = value;
            return self();
        }

        /**
         * @param value the new value
         * @return {@code this}
         * @since 5.5
         */
        @Nonnull
        public Builder strategyId(@Nullable String value) {
            strategyId = value;
            return self();
        }

        @Nonnull
        @Override
        protected Builder self() {
            return this;
        }
    }
}
