package com.atlassian.bitbucket.concurrent;

import javax.annotation.Nonnull;
import java.io.Serializable;
import java.util.concurrent.TimeUnit;

/**
 * Executor that groups submitted tasks in 'buckets' and asynchronously processes these buckets of tasks on worker
 * threads. At {@link ConcurrencyService#getBucketedExecutor(String, BucketedExecutorSettings) construction time}, a
 * {@link BucketedExecutorSettings} is provided that defines the maximum concurrency level, the task bucketing logic
 * and the {@link BucketProcessor} that's called for processing buckets of tasks.
 *
 * @param <T> the task type
 */
public interface BucketedExecutor<T extends Serializable> {

    /**
     * Add a task to be processed and schedule the {@link BucketProcessor} to process the task's bucket with a
     * {@code delay}. The task may still be processed earlier if a prior task for the same bucket has been scheduled
     * to run before the scheduled time.
     *
     * @param task the task to submit
     * @param delay the processing delay
     * @param timeUnit the delay's timeUnit
     */
    void schedule(@Nonnull T task, long delay, @Nonnull TimeUnit timeUnit);

    /**
     * Shuts down the executor and releases all resources acquired by the executor. This shuts down the executor on the
     * current node only. In a cluster, the corresponding executors will keep processing tasks until they're shut down.
     * <p>
     * <strong>Note:</strong> All plugins which {@link ConcurrencyService#getBucketedExecutor request} a bucketed
     * executor <strong>must</strong> shut it down when the plugin is disabled. Failing to do so will leave behind
     * orphaned instances. On plugin upgrade this can be particularly problematic given the fact that when the new
     * version of the plugin requests an executor it will be provided with the old stale instance which may trigger
     * unexpected behavior, like {@link ClassCastException}s caused by {@link ClassLoader} mismatches. This behaviour
     * will continue until the next time the system restarts (in a clustered environment this means every node).
     */
    void shutdown();

    /**
     * Submit a task for processing and schedule the {@link BucketProcessor} to process the task's bucket immediately.
     * This is equivalent to calling {@link #schedule(Serializable, long, TimeUnit)} with a {@code delay} of {@code 0}.
     * @param task the task to submit
     */
    void submit(@Nonnull T task);
}
