package com.atlassian.bitbucket.concurrent;

import javax.annotation.Nonnull;
import java.io.Serializable;
import java.util.List;

/**
 * {@link BucketedExecutor} processor that's called when a bucket of tasks is ready for processing. Implementations
 * need to ensure that the processing of a bucket of tasks completes or fails as a whole; when
 * {@link #process(String, List)} fails and the {@link BucketedExecutor} has been configured to
 * {@link BucketedExecutorSettings#getMaxAttempts() retry} processing, the whole bucket will be retried at a later
 * time.
 *
 * @param <T> the task type
 */
public interface BucketProcessor<T extends Serializable> {

    /**
     * Callback method for processing a 'bucket' of tasks. Implementations need to ensure that processing of a bucket
     * of tasks succeeds or fails as a whole. If the {@code process} method throws an exception and the executor has
     * been configured to retry processing, all items in {@code tasks} will be re-offered at a later time.
     *
     * @param bucketId the bucket ID as determined by the configured
     *                 {@link BucketedExecutorSettings#getBucketIdExtractor() ID extractor function}.
     * @param tasks    the tasks that are ready for processing (at most {@link BucketedExecutorSettings#getBatchSize()
     *                 batch size} items.
     */
    void process(@Nonnull String bucketId, @Nonnull List<T> tasks);
}
