package com.atlassian.bitbucket.commit;

import com.atlassian.bitbucket.property.PropertyMap;
import com.atlassian.bitbucket.property.SimplePropertySupport;
import com.atlassian.bitbucket.repository.Repository;
import com.atlassian.bitbucket.user.Person;
import com.google.common.collect.ImmutableList;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Objects;

import static com.google.common.base.MoreObjects.firstNonNull;
import static java.util.Objects.requireNonNull;

public class SimpleCommit extends SimpleMinimalCommit implements Commit, Serializable {

    private final Person author;
    private final Date authorTimestamp;
    private final Person committer;
    private final Date committerTimestamp;
    private final String message;
    private final List<MinimalCommit> parents;

    private transient SimplePropertySupport propertySupport;
    private transient Repository repository;

    private SimpleCommit(Builder builder) {
        super(builder.minimalCommitBuilder);

        //Fallback on the committer details if author details weren't set, and throw if neither was set
        //This doesn't use firstNonNull because that throws on its own if both arguments are null, and
        //doesn't allow specifying a message like requireNonNull does
        author = requireNonNull(builder.author == null ? builder.committer : builder.author, "author");
        authorTimestamp = requireNonNull(
                builder.authorTimestamp == null ? builder.committerTimestamp : builder.authorTimestamp,
                "authorTimestamp");
        //Default the committer details to the author details if they weren't set explicitly. By this
        //point, we know one or the other was provided
        committer = firstNonNull(builder.committer, builder.author);
        committerTimestamp = firstNonNull(builder.committerTimestamp, builder.authorTimestamp);
        message = builder.message;
        parents = ImmutableList.copyOf(builder.parents);
        propertySupport = new SimplePropertySupport(builder);
        repository = builder.repository;
    }

    @Override
    public boolean equals(Object o) {
        if (super.equals(o) && o instanceof SimpleCommit) {
            SimpleCommit that = (SimpleCommit) o;

            return Objects.equals(getAuthor(), that.getAuthor()) &&
                    Objects.equals(getAuthorTimestamp(), that.getAuthorTimestamp()) &&
                    Objects.equals(getMessage(), that.getMessage());
        }
        return false;
    }

    @Nonnull
    @Override
    public Person getAuthor() {
        return author;
    }

    @Nonnull
    @Override
    public Date getAuthorTimestamp() {
        return authorTimestamp;
    }

    @Nonnull
    @Override
    public Person getCommitter() {
        return committer;
    }

    @Nonnull
    @Override
    public Date getCommitterTimestamp() {
        return committerTimestamp;
    }

    @Override
    public String getMessage() {
        return message;
    }

    @Nonnull
    @Override
    public List<MinimalCommit> getParents() {
        return parents;
    }

    @Nonnull
    @Override
    public PropertyMap getProperties() {
        return propertySupport.getProperties();
    }

    @Override
    public Repository getRepository() {
        return repository;
    }

    @Override
    public int hashCode() {
        return Objects.hash(getId(), getAuthor(), getAuthorTimestamp(), getMessage());
    }

    public void setProperties(@Nonnull PropertyMap properties) {
        propertySupport.setProperties(properties);
    }

    public void setRepository(Repository repository) {
        if (this.repository != null) {
            throw new IllegalStateException("Repository has already been set; you cannot override once set.");
        }
        this.repository = repository;
    }

    @Override
    public String toString() {
        return "SimpleCommit{id='" + getId() +
                "', author='" + getAuthor() +
                "', authorTimestamp=" + getAuthorTimestamp() +
                "', committer='" + getCommitter() +
                "', committerTimestamp=" + getCommitterTimestamp() +
                ", message='" + getMessage() + "'}";
    }

    public static final class Builder extends SimplePropertySupport.AbstractPropertyBuilder<Builder> {

        private final SimpleMinimalCommit.Builder minimalCommitBuilder;
        
        private Person author;
        private Date authorTimestamp;
        private Person committer;
        private Date committerTimestamp;
        private String message;
        private List<MinimalCommit> parents;
        private Repository repository;

        public Builder(@Nonnull String id) {
            minimalCommitBuilder = new SimpleMinimalCommit.Builder(id);
            parents = new ArrayList<>();
        }

        public Builder(@Nonnull Commit commit) {
            super(commit);

            author = commit.getAuthor();
            authorTimestamp = commit.getAuthorTimestamp();
            committer = commit.getCommitter();
            committerTimestamp = commit.getCommitterTimestamp();
            message = commit.getMessage();
            minimalCommitBuilder = new SimpleMinimalCommit.Builder(commit);
            parents = new ArrayList<>(commit.getParents());
            repository = commit.getRepository();
        }

        /**
         * Sets the {@link Commit#getAuthor() author}. If the {@link #committer(Person) committer} has not been set,
         * the provided {@link Person person} will also be set as the committer.
         *
         * @param value the author to associate with the commit
         * @return {@code this}
         */
        @Nonnull
        public Builder author(@Nonnull Person value) {
            author = requireNonNull(value, "author");

            return self();
        }

        /**
         * Sets the {@link Commit#getAuthorTimestamp() author timestamp}. If the provided {@code Date} is {@code null},
         * the current date is used. If the {@link #committerTimestamp(Date) committer timestamp} has not been set, the
         * provided date (or the current date) will also be set as the committer timestamp.
         *
         * @param value the author timestamp to associate with the commit, or {@code null} to use the current date
         * @return {@code this}
         */
        @Nonnull
        public Builder authorTimestamp(@Nullable Date value) {
            authorTimestamp = value == null ? null : new Date(value.getTime());

            return self();
        }

        @Nonnull
        public SimpleCommit build() {
            return new SimpleCommit(this);
        }

        @Nonnull
        public Builder clearParents() {
            parents.clear();

            return self();
        }

        /**
         * Sets the {@link Commit#getCommitter() committer}. If the {@link #author(Person) author} has not been set,
         * the provided {@link Person person} will also be set as the author.
         *
         * @param value the committer to associate with the commit
         * @return {@code this}
         * @since 5.0
         */
        @Nonnull
        public Builder committer(@Nonnull Person value) {
            committer = requireNonNull(value, "committer");

            return self();
        }

        /**
         * Sets the {@link Commit#getCommitterTimestamp() committer timestamp}. If the provided {@code Date} is
         * {@code null}, the current date is used. If the {@link #authorTimestamp(Date) author timestamp} has not
         * been set, the provided date (or the current date) will also be set as the author timestamp.
         *
         * @param value the committer timestamp to associate with the commit, or {@code null} to use the current date
         * @return {@code this}
         * @since 5.0
         */
        @Nonnull
        public Builder committerTimestamp(@Nullable Date value) {
            committerTimestamp = value == null ? null : new Date(value.getTime());

            return self();
        }

        @Nonnull
        public Builder displayId(@Nullable String value) {
            minimalCommitBuilder.displayId(value);
            
            return self();
        }

        @Nonnull
        public Builder message(@Nullable String value) {
            message = value;

            return self();
        }

        @Nonnull
        public Builder parent(@Nullable MinimalCommit value) {
            addIf(Objects::nonNull, parents, value);

            return self();
        }

        @Nonnull
        public Builder parents(@Nullable MinimalCommit value, @Nullable MinimalCommit... values) {
            addIf(Objects::nonNull, parents, value, values);

            return self();
        }

        @Nonnull
        public Builder parents(@Nullable Iterable<MinimalCommit> values) {
            addIf(Objects::nonNull, parents, values);

            return self();
        }

        @Nonnull
        public Builder repository(@Nullable Repository value) {
            repository = value;

            return self();
        }

        @Nonnull
        @Override
        protected Builder self() {
            return this;
        }
    }
}
