package com.atlassian.bitbucket.build.status;

import com.atlassian.bitbucket.build.BuildOrder;
import com.atlassian.bitbucket.repository.Repository;
import com.atlassian.bitbucket.util.BuilderSupport;
import org.apache.commons.lang3.StringUtils;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import static com.google.common.base.MoreObjects.firstNonNull;
import static java.util.Objects.requireNonNull;
import static org.apache.commons.lang3.StringUtils.stripToNull;

/**
 * A request class containing the base information all build status searches can use
 *
 * @since 7.6
 */
public abstract class AbstractBuildStatusSearchRequest {

    private final BuildOrder buildOrder;
    private final String commitId;
    private final String key;
    private final Repository repository;

    protected AbstractBuildStatusSearchRequest(@Nonnull AbstractBuilder<?> builder) {
        buildOrder = firstNonNull(builder.buildOrder, BuildOrder.NEWEST);
        commitId = builder.commitId;
        key = builder.key;
        repository = builder.repository;
    }

    /**
     * The order to return build statuses in.
     *
     * @return the {@link BuildOrder} to sort the results.
     */
    @Nonnull
    public BuildOrder getBuildOrder() {
        return buildOrder;
    }

    /**
     * Restrict the search to build statuses associated with this commit ID.
     *
     * @return The commit ID to restrict the search, or null to search across all commits.
     */
    @Nullable
    public String getCommitId() {
        return commitId;
    }

    /**
     * Restrict the search to build statuses associated with this key.
     *
     * @return The key to restrict the search or null to search across all keys.
     */
    @Nullable
    public String getKey() {
        return key;
    }

    /**
     * The repository for the build statuses
     *
     * @return The repository that the build statuses are associated with
     */
    @Nonnull
    public Repository getRepository() {
        return repository;
    }

    public abstract static class AbstractBuilder<T extends AbstractBuilder<T>> extends BuilderSupport {

        private final Repository repository;
        private BuildOrder buildOrder;
        private String commitId;
        private String key;

        public AbstractBuilder(@Nonnull Repository repository) {
            this.repository = requireNonNull(repository, "repository");
        }

        public AbstractBuilder(@Nonnull AbstractBuildStatusSearchRequest request) {
            requireNonNull(request);
            commitId = request.getCommitId();
            repository = request.getRepository();
            buildOrder = request.getBuildOrder();
            key = request.getKey();
        }

        public abstract AbstractBuildStatusSearchRequest build();

        @Nonnull
        public T buildOrder(@Nonnull BuildOrder buildOrder) {
            this.buildOrder = buildOrder;

            return self();
        }

        @Nonnull
        public T commitId(@Nullable String commitId) {
            this.commitId = stripToNull(commitId);

            return self();
        }

        @Nonnull
        public T key(@Nullable String key) {
            this.key = StringUtils.trimToNull(key);

            return self();
        }

        @Nonnull
        public abstract T self();
    }
}
