package com.atlassian.bitbucket.pull;

import com.atlassian.bitbucket.comment.Comment;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import static java.util.Objects.requireNonNull;

/**
 * Describes a request to {@link PullRequestService#finishReview(PullRequestFinishReviewRequest) finish a review}
 * on a {@link PullRequest}.
 *
 * @since 7.7
 */
public class PullRequestFinishReviewRequest {

    private final String commentText;
    private final PullRequestParticipantStatus participantStatus;
    private final PullRequest pullRequest;

    private PullRequestFinishReviewRequest(@Nonnull Builder builder) {
        commentText = builder.commentText;
        participantStatus = builder.participantStatus;
        pullRequest = builder.pullRequest;
    }

    /**
     * Get the comment text that will be used for a {@link Comment general comment} that is added to the
     * {@link PullRequest pull request} when you complete a review.
     *
     * @return the comment text or {@code null} if there is no comment text
     */
    @Nullable
    public String getCommentText() {
        return commentText;
    }

    /**
     * Get the {@link PullRequestParticipantStatus participant status} to be used as the status for a reviewer's status
     * on a {@link PullRequest pull request} when you complete a review.
     *
     * @return the participant status or {@code null} if there is no participant status
     */
    @Nullable
    public PullRequestParticipantStatus getParticipantStatus() {
        return participantStatus;
    }

    /**
     * Get the {@link PullRequest pull request} that is used for the review.
     *
     * @return the pull request
     */
    @Nonnull
    public PullRequest getPullRequest() {
        return pullRequest;
    }

    public static class Builder {

        private final PullRequest pullRequest;

        private String commentText;
        private PullRequestParticipantStatus participantStatus;

        public Builder(@Nonnull PullRequest pullRequest) {
            this.pullRequest = requireNonNull(pullRequest, "pullRequest");
        }

        @Nonnull
        public PullRequestFinishReviewRequest build() {
            return new PullRequestFinishReviewRequest(this);
        }

        @Nonnull
        public Builder commentText(@Nullable String value) {
            commentText = value;
            return this;
        }

        @Nonnull
        public Builder participantStatus(@Nullable PullRequestParticipantStatus value) {
            participantStatus = value;
            return this;
        }
    }
}
